/* ************************************************************
  Postprocessor for oscillating tangential knives (OTK)
  Modifies a given engraving G-Code and adds special commands 
  for using a tangential knife                                 
 
  Short description how it works:
  The program first reads a specified G-Code file and then 
  parses each line to extract the information needed for the
  tangential knife calculations. Then these entries are processed 
  to add the necessary knife rotations. 
  If the knife rotation exceeds a parameterized limit the knife 
  is first retracted, then rotated and moved back to continuing 
  the cut. Also the safe z position for this operation can be
  specified.
  The default axis of the knife can be changed by a command 
  option and additional comments can be added for debugging
  or understanding purpose.
  Optionally, the calculated knife angles can be restricted to 
  be in the range of [0..360). This may be used for those 
  machines that allows rotations modulo 360 and directly uses
  the shortest way (e.g. if the machine only rotates 1" when 
  moving from 359° to 0° instead of 359° backward). In most
  cases the default settings (no modulo) should be correct.
  The modified G-Code is printed to stdout and may be piped
  into a target file.
  Please note: 
   - The program assumes metric values for default feed and z-retract. 
     If the machine uses imperial units (inch) you should adjust
     the values or pass the values as parameters.
   - The program cannot handle GCode that includes variables 

  Compile the postprocessor by calling make or simply 
    g++ OTKPP.cc -o OTKPP
  ************************************************************ */

#include <algorithm>
#include <fstream>
#include <iostream>
#include <iomanip>
#include <math.h>
#include <string>
#include <sstream>
#include <vector>


using namespace std;


const int  CommentRow    = 40;    // Start column of comments
const int  Digits        =  3;    // Output digits of added values
const bool StopOnRetract = false; // stop ozcillation on retract 


#define PI 3.14159265;
#define max(a,b) (((a) >= (b))? (a):(b))


enum CmdType   // Relevant types of G-Code commands
{
  OTHER = 0,   // Nothing to do (not relevant for OTG)
  FAST,        // G0 fast move used to detect initialization
  LINE,        // G1 straight move 
  ARC_CW,      // G2 clockwise arc
  ARC_CCW,     // G3 counter-clockwise arc
  MOTOR_ON,    // Motor on used to reset knife angle
  MOTOR_OFF    // Motor off
};


// The necessary information of one G-Code command line
class CENTRY 
{
  public:
    CENTRY () = default;
    CENTRY (CmdType T, const string &L,
	    double X, double Y, double Z, double F,
	    double A);
    virtual ~CENTRY() = default;
  
    CmdType Type       = OTHER;
    string  Line       = {};
    double  PosX       = 0.0;
    double  PosY       = 0.0;
    double  PosZ       = 0.0;
    double  FeedZ      = 0.0;
    double  StartAngle = 0.0;
    double  EndAngle   = 0.0;
    double  CenterX    = 0.0;
    double  CenterY    = 0.0;
};


/*! \class The main class of the OTK preprocessor. Contains all 
           necessary settings, objects and methods 
*/
class COTKPP
{
  public:
    COTKPP (int argc, char *argv[]);
    virtual ~COTKPP ();

    int Run ();
  
  private:
    string  StrToLower (string s) const;
    string  StrToUpper (string s) const;
    string  TypeToStr (CmdType Type) const;
    double  CalcAngleDiff (double Angle1, double Angle2) const;
    double  GetNearestAngle (double LastAngle, double Angle) const;
    double  RoundAngle (double Angle, unsigned int Digits) const;
    void    AddLine (const string& CommandStr, const string& CommentStr) const;
    string  RemoveComments (const string& Line) const;
    double  GetValue (const string& Line, size_t Pos) const;
    void    GetLinAngles (double x1, double y1, double x2, double y2,
                          double LastAngle, double &Angle1, double &Angle2) const;
    void    GetArcAngles (double x1, double y1, double x2, double y2,
	 		  double i, double j, bool orient,
		          double LastAngle, double &Angle1, double &Angle2) const;
    CmdType ParseCmd (const string &Line,
    	              double &X, double &Y, double &Z,
		      double &CX, double &CY, double &F) const;
    double  SetKnifeAngle (double OldAngle, double NewAngle,
 		           double Feed, double PosZ, bool NoRetract = false) const;
    bool    EvaluateArguments (int argc, char *argv[]);
    string  GetCommentStr (const CENTRY& Entry,
			   double Angle, CmdType LastType) const;
    void    ReadFile ();
    void    PrintEntry (const CENTRY &Entry) const;
    void    ProcessData ();

    //  Just some methods for testing: 
    //  void Test () const;
    //  void Test (double Angle1, double Angle2, double Expected) const;
  
    // The Modulo setting depends on the machine configuration.
    // The value should be set to false if the machines doesn't handle
    // the angles modulo (e.g. rotates about 359 degree when rotation is set
    // from 359° to 0°). May depend on wrapped rotary settings.
    bool   Modulo   = false;  // Use modulo for angles (range [0..360)°)
    bool   Comments = false;  // Comment the G-Code lines
    char   Axis     =   'C';  // Default axis of oscillating knife
    double Limit    =  22.5;  // Max. rotation angle without retract
    double SafeZ    =   2.0;  // Safety height for retract [assumed mm]
    double FeedZ    = 200.0;  // Default z-feed after retract (if not found in program
                              // (assume mm/min)
    double Dwell    =   3.0;  // Seconds to wait for oszillation runup
  
    string InFile{};          // No default filename
    string OutFile{};         // Nmae of the output file, stdout if not defined
  
    vector<CENTRY> Data;
    ifstream       In{};
    streambuf*     CinBuf{nullptr};
    ofstream       Out{};
    streambuf*     CoutBuf{nullptr};
};


/*!
  \brief Constructor of the CENTRY class

  \param[in]  T  Type of the entry
  \param[in]  L  String of the command line
  \param[in]  X  X coordinate of the entry
  \param[in]  Y  Y coordinate of the entry
  \param[in]  Z  Z coordinate of the entry
  \param[in]  F  Feed in z direction
  \param[in]  A  Rotation angle that belongs
*/
CENTRY::CENTRY (CmdType T, const string& L, 
		double X, double Y, double Z, double F,
	        double A) :
  Type(T),
  Line(L),
  PosX(X),
  PosY(Y),
  PosZ(Z),
  FeedZ(F),
  StartAngle(A),
  EndAngle(A)
{}


/*!
  \brief Constructor of the OTK post processor class

  \param[in]  argc  Number of program parameters
  \param[in]  argv  List of the parameters
*/
COTKPP::COTKPP (int argc, char *argv[])
{
  if (! EvaluateArguments(argc, argv))
    exit (1);
} 


/*!
  \brief Destructuor of the OTK post processor class

  Closes open file handles
*/
COTKPP::~COTKPP ()
{
  if (! InFile.empty())
    In.close();

  if (! OutFile.empty())
    Out.close();
}


/*! 
  \brief Converts a string to lower case

  \param[in]  s  String to be modified

  \return Converted lower-case string
*/
string COTKPP::StrToLower (string s) const
{
  transform(s.begin(), s.end(), s.begin(),
            [](unsigned char c){ return std::tolower(c); }
	   );
  return s;
}


/*!
  \brief Converts a string to upper case

  \param[in]  s  String to be modified

  \return Converted upper-case string
*/
string COTKPP::StrToUpper (string s) const
{
  transform(s.begin(), s.end(), s.begin(),
            [](unsigned char c){ return std::toupper(c); }
	   );
  return s;
}


/*!
  \brief returns a readable string of the given command type

  \param[in]  Type  Type of requested string

  \return Name of the given type
*/
string COTKPP::TypeToStr (CmdType Type) const
{
  string Result;

  switch (Type)
  {
    case FAST:
      Result = "FAST";
      break;
    case LINE:
      Result = "LINE";
      break;
    case ARC_CW:
      Result = "ARC_CW";
      break;
    case ARC_CCW:
      Result = "ARC_CCW";
      break;
    case MOTOR_ON:
      Result = "MOTOR ON";
      break;
    case MOTOR_OFF:
      Result = "MOTOR OFF";
      break;      
    default:
      Result = "OTHER";
  }

  return Result;
}


/*!
   \brief Calculats the difference of two angles

   Returns the smallest signed difference of the given 
   Angles modulo 360.0. The Result is in the range of
   [-180..180).

  \param[in]  Angle1  First angle
  \param[in]  Angle2  Second angle

  \return Difference of both directions
*/
double COTKPP::CalcAngleDiff (double Angle1, double Angle2) const
{
  return fmod(Angle2 - Angle1 + 360000000 + 180.0, 360.0) - 180.0;
}


/*!
   \brief Get the nearest angle regarding the current orientation

   Returns the nearest angle according to the given 
   last angle. For example, if the last angle is 180.0 degree
   and the new Angle is -90.0 degree, this would be an absolute
   rotation of -270 degree. Because -90° is the same as 270 degree
   the result would be 270 degree which needs only a positive 
   rotation of 90 degree.

  \param[in]  LastAngle  Reference angle
  \param[in]  Angle      Angle to calculate the nearest angle for

  \return Angle with the smallest change of the given LastAngle
*/
double COTKPP::GetNearestAngle (double LastAngle, double Angle) const
{
  return LastAngle + CalcAngleDiff (LastAngle, Angle);
}


/*!
  \brief Rounds a value to a number of digits

  Rounds the angle value to a given digit and (if set)
  calculates the modulo of value to be in the range of
  [0..360)

  \param[in]  Angle   Angle value to be processed
  \param[in]  Digits  Amount of relevant digits 

  \return Rounded angle value
*/
double COTKPP::RoundAngle (double Angle, unsigned int Digits) const
{
  const double Factor = pow (10, Digits);

  if (Modulo)
  {
    Angle = round((Angle + 360.0) * Factor) / Factor;
    return fmod (Angle, 360.0);
  }
  else
    return round(Angle * Factor) / Factor;
}


/*!
  \brief Adds an output line of command and comment

  Adds a possibly modified command line to the output and
  also adds a comment if desired.

  \param[in]  Command  Command string to be printed
  \param[in]  Comment  Comment string according to the command
*/
void COTKPP::AddLine (const string& CommandStr, const string& CommentStr) const
{
  cout << CommandStr;

  if(Comments && CommentStr.length() > 0)
  {
    int count = CommentRow - CommandStr.length();
    count = max (2, count);
    
    const string Spaces (count, ' ');
    cout << Spaces << "( " << CommentStr << " )";
  }
  
  cout << endl;
}
  

/*!
  \brief Removes G-Code comment(s) a given line

  Removes the G-Code comments from the given line.
  This is used for preprocessing the lines before 
  parsing them to avoid handling of possible 
  commands inside the comments.

  \param[in]  Line  Command string to be processed

  \return Command string without comments
*/
string COTKPP::RemoveComments (const string& Line) const
{
  string Result = "";
  
  size_t i = 0;
  while (i < Line.length())
  {
    if (Line[i] == '(')
      while (i < Line.length() && i != ')')
        i++;
    else if (Line[i] != '\n')
      Result += Line[i];
    i++;
  }
  return Result;
}
  

/*!
  \brief Returns a G-Code parameter values at a given position

  Returns the G-Code command parameter at the given 
  position of a line.

  \param[in]  Line  Command string to be evaluated
  \param[in]  Pos   Position to start reading the value

  \return Value of the parameter
*/ 
double COTKPP::GetValue (const string& Line, size_t Pos) const
{
  double Result;
    
  size_t End = Line.find (' ', Pos);
  if (End == string::npos)
    Result = atof (Line.substr(Pos+1).c_str());
  else
    Result = atof (Line.substr(Pos+1, End-Pos-1).c_str());

  return Result;
}


/*!
  \brief Calculates the orientation of a linear move

  Returns the knife angle for a given linear (G1) command
  The function returns the nearest angle <= 180° relative
  to the given last angle. E.g. if the last angle is 200°
  and the move has a calculated angle of -90°, then the 
  result would be 270° instead of -90°.
  The splitted angle is used to simplyfy the assignment 
  of the return value.

  \param[in]       x1          Start (current) x position 
  \param[in]       y1          Start (current) y position
  \param[in]       x2          End x-position of the move
  \param[in]       y2          End y-position of the move
  \param[in]       LastAnagle  Last reference angle
  \param[in, out]  Angle1      Start angle
  \param[in, out]  Angle2      End end (always same as Angle1)
*/
void COTKPP::GetLinAngles (double x1, double y1, double x2, double y2,
			   double LastAngle, double &Angle1, double &Angle2) const
{
  double Angle = atan2 (y2-y1, x2-x1) * 180.0 / PI;
  Angle = GetNearestAngle (LastAngle, Angle);

  Angle = RoundAngle (Angle, Digits);

  Angle1 = Angle;
  Angle2 = Angle;
}


/*!
  \brief Calculates the start- and end-rotation of an arc move

  Returns the start and end angle of a given arc command
  The returned angles are the nearest angles (see GetLinAngles
  for description.

  \param[in]       x1          Start (current) x position 
  \param[in]       y1          Start (current) y position
  \param[in]       x2          End x-position of the move
  \param[in]       y2          End y-position of the move
  \param[in]       i           relative x position of the center
  \param[in]       j           relative y position of the center
  \param[in]       orient      Arc orientation (false = CCW, true = CW)
  \param[in]       LastAnagle  Last reference angle
  \param[in, out]  Angle1      Start angle
  \param[in, out]  Angle2      End end (always same as Angle1)
*/
void COTKPP::GetArcAngles (double x1, double y1, double x2, double y2,
			   double i, double j, bool orient,
		           double LastAngle, double &Angle1, double &Angle2) const
{
  // Calculate the tangent angle at the start position
  Angle1 = atan2(-j, -i) * 180.0 / PI;
  Angle1 += orient? -90.0:90.0;
  Angle1 = GetNearestAngle (LastAngle, Angle1);
  Angle1 = RoundAngle (Angle1, Digits);

  // Center relative to end position
  double i2 = (x1 + i) - x2;
  double j2 = (y1 + j) - y2;

  // Calculate the tangent angle at the end position
  Angle2 = atan2(-j2, -i2) * 180.0 / PI;
  Angle2 += orient? -90.0:90.0;
  Angle2 = GetNearestAngle (Angle1, Angle2);
  Angle2 = RoundAngle (Angle2, Digits);
}


/*!
   \brief Parses a single G-Code command

   Evaluates the given command line and returns the command
   type, the x, y and z position, the relative position of 
   the center for arcs and the feed. 
   Only the found values are changed. The passed Values of 
   missing values keep unchanged. So, the last values are 
   simply passed and only the existing parameters are updated.

  \param[in]       Line  G-code command line to be parsed
  \param[in, out]  X     New (extracted) x coordinate
  \param[in, out]  Y     New (extracted) y coordinate
  \param[in, out]  Z     New (extracted) z coordinate
  \param[in, out]  CX    Relative x offset of arc center
  \param[in, out]  CY    Relative y offset of arc center
  \param[in, out]  F     New (extracted) feed value

  \return Extracted command type of the given G-Code line
*/
CmdType COTKPP::ParseCmd (const string &Line,
    	                  double &X, double &Y, double &Z,
		          double &CX, double &CY, double &F) const
{
  CmdType Result = OTHER;

  if ((Line.find("G0 ") != string::npos)  ||
      (Line.find("G00") != string::npos))
    Result = FAST;
  else if ((Line.find("G1 ") != string::npos)  ||
           (Line.find("G01") != string::npos))
    Result = LINE;
  else if ((Line.find("G2 ") != string::npos)  ||
	   (Line.find("G02") != string::npos))
    Result = ARC_CW;
  else if ((Line.find("G3 ") != string::npos)  ||
 	   (Line.find("G03") != string::npos))
    Result = ARC_CCW;
  else if (((Line.find("M3") != string::npos) &&
	    (Line.find("M30") == string::npos)) ||
	   (Line.find("M03") != string::npos))
    Result = MOTOR_ON;
  else if ((Line.find("M5") != string::npos) ||
	   (Line.find("M05") != string::npos))
    Result = MOTOR_OFF;
	   
  // Extract necessary coordinates of the move commands
  if (Result != OTHER && Result != MOTOR_ON && Result != MOTOR_OFF)
  {
    size_t PosX = Line.find ("X");
    size_t PosY = Line.find ("Y");
    size_t PosZ = Line.find ("Z");
    size_t PosF = Line.find ("F");
    size_t PosI = Line.find ("I");
    size_t PosJ = Line.find ("J");

    if (PosX != string::npos)
      X = GetValue (Line, PosX);
    if (PosY != string::npos)
      Y = GetValue (Line, PosY);
    if (PosZ != string::npos)
    {
      Z = GetValue (Line, PosZ);
      if (PosF != string::npos)
        F = GetValue (Line, PosF);  // Only store z feed
    }
    if (PosI != string::npos)
      CX = GetValue (Line, PosI);
    if (PosJ != string::npos)
      CY = GetValue (Line, PosJ);
    
    if ((PosX == string::npos && PosY == string::npos &&
	 PosZ == string::npos))
      Result = OTHER;
  }

  return Result;
}


/*!
  \brief Sets the orientation of the knife

  Sets the knife angle. Checks whether the angle is even changed
  and a retraction is needed because the max rotation limit is
  exceeded. In this case (and if the NoRectract is not set, the
  knive will be moved out of the material before the rotation.
  Returns the new angle.

  \param[in]  OldAngle   Last angle (used for comparion)
  \param[in]  NewAngle   Angle to rotate the knife to
  \param[in]  Feed       Z-feed to return retract move
  \param[in]  PosZ       Z-position to return after retract
  \param[in]  NoRetract  Disables the rectraction if true

  \return New knife angle (just used for easy assignment)
*/
double COTKPP::SetKnifeAngle (double OldAngle, double NewAngle,
			      double Feed, double PosZ, bool NoRetract /*=false*/) const
{
  if (OldAngle == NewAngle)
    return NewAngle;
  
  stringstream CmdStr;
  CmdStr.precision (Digits);

  const double Rotation = CalcAngleDiff (OldAngle, NewAngle);
  const bool retract = (! NoRetract && fabs(Rotation) > Limit);

  if (retract)
  {
    CmdStr << "G0 Z" << fixed << SafeZ;
    AddLine (CmdStr.str(), "Retract for knife rotation");
  }

  if (StopOnRetract)
    AddLine ("M5", "Stop oscillation during rotation");

  CmdStr.str ("");
  CmdStr << "G0 " << Axis << fixed << NewAngle;
  AddLine (CmdStr.str(), "Rotate knife " + to_string(Rotation) + "°");

  if (StopOnRetract)
  {
    AddLine ("M3", "Restart oscillation after rotation");
    if (Dwell > 0)
      AddLine ("G4 P" + to_string(Dwell), "Runup oszillation");
  }

  if (retract)
  {
    CmdStr.str ("");
    CmdStr << "G1 F" << Feed << " Z" << fixed << PosZ;
    AddLine (CmdStr.str(), "Restore z position");
  }

  return NewAngle;
}


/*!
  \brief Check the passed programm arguments

  Evaluates the program paramters and overwrites the default
  settings by the specified values. If an error was found or
  no filename is specified an syntax messages is printed.

  \param[in]  argc  Number of program parameters
  \param[in]  argv  List of the parameters

  \return true, if the parameters are valid, false otherwise
*/
bool COTKPP::EvaluateArguments (int argc, char *argv[]) 
{
  bool Error = false;
  
  int i = 1;
  while (i < argc)
  {
    string Option = StrToLower(argv[i++]);
    
    if ((Option == "-a" || Option == "--axis") && i < argc)
      Axis = argv[i++][0];
    else if (Option == "-dbg" || Option == "--debug")
      Comments = true;
    else if ((Option == "-d" || Option == "--dwell") && i < argc)
      Dwell = atof (argv[i++]);
    else if ((Option == "-f" || Option == "--feed") && i < argc)
      FeedZ = atof (argv[i++]);
    else if ((Option == "-i" || Option == "--in") && i < argc)
      InFile = argv[i++];
    else if ((Option == "-l" || Option == "--limit") && i < argc)
      Limit = atoi (argv[i++]);
    else if (Option == "-m" || Option == "--modulo")
      Modulo = true;
    else if ((Option == "-o" || Option == "--out") && i < argc)
      OutFile = argv[i++];
    else if ((Option == "-z" || Option == "--zsafe") && i < argc)
      SafeZ = atof (argv[i++]);
    else if (Option == "-?" || Option == "-h" || Option == "--help")
      Error = true;
  }

  if (argc == 1 || Error)
  {
    cout << ((argc == 1)?"No":"Unknown") << " option specified!" << endl;
    cout << "Syntax: " << argv[0] << " options" << endl;
    cout << "  Options:" << endl;
    cout << "  -a   --axis axisname   Name of the knife axis (default: " <<
            Axis << ")" << endl;
    cout << "  -d   --dwell delay     Delay time [s] to runup oscillation (default : " <<
            Dwell << "sec)" << endl;
    cout << "  -dbg --debug           Adds comments to the output" << endl;
    cout << "  -f   --feed speed      Default z-feed (default: " << FeedZ << ")" << endl;
    cout << "  -h   -? | --help       Prints this message" << endl;
    cout << "  -i   --in filename     Path of input gcode file" << endl;
    cout << "  -l   --limit angle     Max. rotation without retraction (default: " <<
            Limit << "°)" << endl;
    cout << "  -m   --modulo          Created rotation range [0..360)°" << endl;
    cout << "  -o   --out filename    Path of modified output gcode file" << endl;
    cout << "  -z   --zsafe position  Safe retractrion height (default: " <<
            SafeZ << "mm)" << endl;
    return false;
  }

  return true;
}


/*!
  \brief Prints the data of an entry on stdout

  \param[in]  Entry  Entry ot be printed
*/
void COTKPP::PrintEntry (const CENTRY &Entry) const
{
  cout << TypeToStr(Entry.Type) << ":" <<
          " x=" << Entry.PosX << " y=" << Entry.PosY << " z=" << Entry.PosZ <<
          " cx=" << Entry.CenterX << " cy=" << Entry.CenterY <<
          " fz=" << Entry.FeedZ << endl;
}


/*!
  \brief Reads the whole input file

  Reads and evaluates G-Code file. The lines and their parsed
  data are stored and returned in a vector. A prior evaluated
  vector has the advantage, that the following knife angle can
  be detected easier as processing the file directly.
  The results are written into the Data attribute of the class
*/
void COTKPP::ReadFile ()
{
  double PosX  = 0.0;
  double PosY  = 0.0;
  double PosZ  = 0.0;
  double Angle = 0.0;
  double Feed  = FeedZ;

  if (! InFile.empty())
  {
    In.open (InFile);
    CinBuf = cin.rdbuf();
    cin.rdbuf (In.rdbuf());  // Redirect std::cin to InFile
  }
  
  string Line;
  while (getline(cin, Line))
  {
    string NormalizedLine = StrToUpper (Line);
    NormalizedLine = RemoveComments (NormalizedLine);

    CENTRY Entry (OTHER, Line, PosX, PosY, PosZ, Feed, Angle);
    Entry.Type = ParseCmd (NormalizedLine, Entry.PosX, Entry.PosY, Entry.PosZ,
                           Entry.CenterX, Entry.CenterY, Entry.FeedZ);

    // PrintEntry (Entry); // Activate for debugging

    const CmdType Type = Entry.Type;
    if (Type != FAST && (Entry.PosX != PosX || Entry.PosY != PosY))
    {
      
      if (Type == LINE)
	GetLinAngles (PosX, PosY, Entry.PosX, Entry.PosY, 
         	      Angle, Entry.StartAngle, Entry.EndAngle);
      else if (Type == ARC_CW || Type == ARC_CCW)
        GetArcAngles (PosX, PosY, Entry.PosX, Entry.PosY,
      	              Entry.CenterX, Entry.CenterY, (Type == ARC_CW),
	              Angle, Entry.StartAngle, Entry.EndAngle);
      Angle = Entry.EndAngle;
    }

    Data.emplace_back (Entry);

    PosX = Entry.PosX;
    PosY = Entry.PosY;
    PosZ = Entry.PosZ;
    Feed = Entry.FeedZ;
  }
  
  cin.rdbuf (CinBuf);
}


/*!
  \brief Returns a comment string for a given data entry

  Creates a comment string that can be added to the G-Code 
  commands when the debug option was activated.

  \param[in]  Data      Data entry of a G-Code command line
  \param[in]  Angle     Angle to be printed in the comment
  \param[in]  LastType  Type of the Previous command

  \return Comment string 
*/
string COTKPP::GetCommentStr (const CENTRY& Entry,
	  		      double Angle, CmdType LastType) const
{
  if (! Comments)
    return "";
  
  CmdType Type = Entry.Type;
  
  stringstream CommentStr;
  if (Type != OTHER)
  {
    const string TypeStr = TypeToStr (Type);
    const string Spaces (10 - TypeStr.length(), ' ');
    CommentStr << TypeStr << ":" << Spaces;
  }

  if (Type != OTHER)
  {
    CommentStr << " x:" << Entry.PosX << " y:" << Entry.PosY;
    if (Type == ARC_CW || Type == ARC_CCW)
      CommentStr << " c1:" << Entry.StartAngle <<
	            " c2:" << Entry.EndAngle;
    else
      CommentStr << " " << Axis << ":" << Angle;
    if (LastType == FAST && Type != FAST)
      CommentStr << " z:" << Entry.PosZ << " f:" << Entry.FeedZ;
  }

  return CommentStr.str();
}


/*!
  \brief Processes the data of the input G-Code file

  Iterates through the class' data vector and adds the needed 
  rotations of the tangential knife.
*/
void COTKPP::ProcessData ()
{
  CmdType LastType = OTHER;
  double  PosX     =   0.0;
  double  PosY     =   0.0;
  double  PosZ     =   0.0;
  double  Angle    =  -1.0;
  double  Feed     = FeedZ;

  if (! OutFile.empty())
  {
    Out.open (OutFile);
    CoutBuf = cout.rdbuf();
    cout.rdbuf (Out.rdbuf()); // Redirect std::cout to OutFile
  }
  
  for (unsigned int i = 0; i < Data.size(); i++)
  {
    const CENTRY Act = Data[i];
    CmdType Type = Act.Type;

    if (Type == MOTOR_OFF)
      Angle = SetKnifeAngle (Angle, 0.0, Feed, SafeZ, true);
    
    stringstream CmdStr;
    CmdStr.precision (Digits);
    CmdStr << Act.Line;
    
    if (Act.PosX != PosX || Act.PosY != PosY)  // Move in x and/or y direction
    {
      if (Type == LINE)
	Angle = SetKnifeAngle (Angle, Act.StartAngle, Act.FeedZ, Act.PosZ);
      else if (Type == ARC_CW || Type == ARC_CCW)
      {
	SetKnifeAngle (Angle, Act.StartAngle, Act.FeedZ, Act.PosZ);
	CmdStr << " " << Axis << setprecision(4+Digits) << Act.EndAngle;
	Angle = Act.EndAngle;
      }
    }
    else if (Type == LINE && Act.PosZ < PosZ) // z move down
    {
      // Adjust knife rotation before plunging into material
      for (size_t j = i+1; j < Data.size(); j++)
      {
        const CENTRY Next = Data[j];
	if (Next.Type == LINE ||
	    Next.Type == ARC_CW || Next.Type == ARC_CCW)
	{
          Angle = SetKnifeAngle (Angle, Next.StartAngle, Next.FeedZ, Next.PosZ);
          break;
        }
      }
    }

    // Get the corresponding comment string
    const string CommentStr = GetCommentStr (Act, Angle, LastType);

    // Add the line to the output
    AddLine (CmdStr.str(), CommentStr);

    if (Type == MOTOR_ON)
    {
      if (Dwell > 0)
        AddLine("G04 P" + to_string(Dwell), "Oszillating start delay");
      Angle = SetKnifeAngle (Angle, 0.0, Feed, SafeZ, true);
    }
    
    LastType = Type;
    PosX     = Act.PosX;
    PosY     = Act.PosY;
    PosZ     = Act.PosZ;
  }

  cout.rdbuf (CoutBuf);
}


/*!
  \brief Method that controls the process

  This handles the execution of the process.

  \return Program return value: 0=success, otherwise failure
*/
int COTKPP::Run()
{
  ReadFile ();
  
  ProcessData ();

  return 0;
}


#if 0  // Currently deactivated test routines
/*!
  \brief Test for calculating the angles

  This method gets two angles and checks if the calculated
  result fits the passed expected value. If the test failes
  an error message is printed.

  \param[in]  Angle1    1st angle
  \param[in]  Angle1    2nd angle
  \param[in]  Expected  Expected result
*/
void COTKPP::Test (double Angle1, double Angle2, double Expected) const
{
  const bool ExpRes = (fabs(Expected) <= Limit);
  cout << "Test " << Angle1 << " / " << Angle2 <<
    "  exp. " << Expected << " (=" << ExpRes << "):" << endl;
  
  const double Diff = CalcAngleDiff (Angle1, Angle2);
  cout << "  Diff: " << Diff;
  bool result = (CalcAngleDiff(Angle1, Angle2) <= Limit);
  
  cout << "  Result = " << result;
  if (Expected != Diff)
    cout << "  ############# DIFF!";  
  if (result != ExpRes)
    cout << "  ############# FAIL!";  
  cout << endl;
}


/*!
  \brief Just a method to test some calculations incl. modulo
*/
void COTKPP::Test () const
{
  Test (   25.0,    51.0,  26.0);
  Test (   20.0,    -1.0, -21.0);
  Test (   20.0,    -6.0, -26.0);
  Test (  355.0,     5.0,  10.0);
  Test (  350.0,    -5.0,   5.0);
  Test (    5.0, - 370.0, -15.0);
  Test (- 725.0,    18.0,  23.0);
  Test (- 725.0,    31.0,  36.0);
  Test (- 135.0, - 142.0,  -7.0);
  Test (- 135.0, - 167.0, -32.0);  
  Test (  400.0,   770.0,  10.0);
  Test (- 135.0, - 115.0,  20.0);
  Test (- 135.0, - 108.0,  27.0);
  Test (-1023.0,   755.0, -22.0);
  Test (  360.0,   360.0,   0.0);
}
#endif


/* 
  Main program that controls the process

  \param[in]  argc  Number of program parameters
  \param[in]  argv  List of the parameters
  
  \return 0 on success, 1 on error
*/
int main (int argc, char *argv[])
{
  COTKPP OTK (argc, argv);

  int Result = OTK.Run();
  
  return Result;
}
