# EL2564 RGBW LED Driver for LinuxCNC

## Beckhoff EL2564 4-Channel PWM LED Output Terminal

This driver provides full support for the Beckhoff EL2564 EtherCAT module - a 4-channel PWM LED driver for RGBW LEDs with 5-48V DC and up to 4A per channel.

---

## Table of Contents

1. [Overview](#overview)
2. [Hardware Specifications](#hardware-specifications)
3. [Installation](#installation)
4. [Configuration](#configuration)
5. [HAL Pins Reference](#hal-pins-reference)
6. [HAL Parameters Reference](#hal-parameters-reference)
7. [Wiring Guide](#wiring-guide)
8. [Example Configurations](#example-configurations)
9. [Troubleshooting](#troubleshooting)
10. [C Driver API Reference](#c-driver-api-reference)

---

## Overview

The EL2564 is a Common Anode LED driver that switches ground (cathode) for each channel. It provides:

- 4 independent PWM channels (Red, Green, Blue, White)
- 16-bit PWM resolution (0-65535)
- Configurable PWM frequency (1-16000 Hz)
- Hardware gamma correction per channel
- Ramp time (fade effect) per channel
- Master gain control (global brightness)
- Open load detection per channel

---

## Hardware Specifications

| Specification | Value |
|---------------|-------|
| Channels | 4 (RGBW) |
| Output voltage | 5-48V DC |
| Max current per channel | 4A |
| PWM frequency | 1-16000 Hz (adjustable) |
| PWM resolution | 16-bit (0-65535) |
| Connection type | Common Anode (switches ground) |
| Vendor ID | 0x00000002 (Beckhoff) |
| Product Code | 0x0a043052 |
| E-Bus current | 70 mA |

### Terminal Assignment

| Terminal | Description | Function |
|----------|-------------|----------|
| 1 | Output LED 1 | PWM output channel 0 (Red) - switches ground |
| 2 | Output LED 3 | PWM output channel 2 (Blue) - switches ground |
| 3 | LED supply+ | +5...48V power supply for LEDs |
| 4 | LED supply- | GND/0V for LEDs (connect to PSU negative) |
| 5 | Output LED 2 | PWM output channel 1 (Green) - switches ground |
| 6 | Output LED 4 | PWM output channel 3 (White) - switches ground |
| 7 | LED supply+ | +5...48V power supply for LEDs |
| 8 | LED supply- | GND/0V for LEDs (connect to PSU negative) |

---

## Installation

### Prerequisites

- LinuxCNC 2.8 or later with EtherCAT support
- linuxcnc-ethercat installed and working
- EtherCAT master (IgH) properly configured

### Step 1: Copy Driver Files

Copy the driver source file to the linuxcnc-ethercat devices directory:

```bash
sudo cp lcec_el2564.c /path/to/linuxcnc-ethercat/src/devices/
sudo cp EL2564.yml /path/to/linuxcnc-ethercat/src/devices/
```

Typical paths:
- Debian/Ubuntu: `/usr/src/linuxcnc-ethercat/src/devices/`
- From source: `~/linuxcnc-ethercat/src/devices/`

### Step 2: Rebuild linuxcnc-ethercat

```bash
cd /path/to/linuxcnc-ethercat
make clean
make
sudo make install
```

### Step 3: Verify Installation

After installation, the EL2564 should be recognized:

```bash
# List all supported device types
halrun -f - <<< "loadrt lcec; show comp" | grep -i el2564
```

---

## Configuration

### EtherCAT XML Configuration (ethercat-conf.xml)

Add the EL2564 slave to your EtherCAT configuration file:

```xml
<masters>
  <master idx="0" appTimePeriod="1000000" refClockSyncCycles="1">
    <!-- Other slaves... -->
    <slave idx="16" type="EL2564" name="EL2564-1"/>
    <!-- More slaves... -->
  </master>
</masters>
```

**Parameters:**
- `idx`: Slave position in the EtherCAT chain (0-based)
- `type`: Must be `EL2564`
- `name`: Custom name for HAL pins (e.g., `LED`, `RGBW`, `STATUS-LED`)

### INI File Configuration

Add the HAL file to your machine's INI configuration:

```ini
[HAL]
HALFILE = your-main.hal
HALFILE = el2564-led.hal    # Add your LED configuration
```

### Basic HAL Configuration

Create a HAL file for LED control (e.g., `el2564-led.hal`):

```hal
# ========================================
# EL2564 LED Configuration
# ========================================

# Set scaling (1.0 = input value equals percentage)
setp lcec.0.EL2564-1.scale-0 1.0
setp lcec.0.EL2564-1.scale-1 1.0
setp lcec.0.EL2564-1.scale-2 1.0
setp lcec.0.EL2564-1.scale-3 1.0

# Enable all channels
setp lcec.0.EL2564-1.enable-0 1
setp lcec.0.EL2564-1.enable-1 1
setp lcec.0.EL2564-1.enable-2 1
setp lcec.0.EL2564-1.enable-3 1

# Set initial color (e.g., green for "ready")
setp lcec.0.EL2564-1.pwm-0 0      # Red off
setp lcec.0.EL2564-1.pwm-1 100    # Green 100%
setp lcec.0.EL2564-1.pwm-2 0      # Blue off
setp lcec.0.EL2564-1.pwm-3 0      # White off
```

---

## HAL Pins Reference

### Input Pins (Control)

| Pin | Type | Description | Range |
|-----|------|-------------|-------|
| `lcec.0.<name>.pwm-0` | float IN | PWM duty cycle channel 0 (Red) | 0.0 - 100.0 |
| `lcec.0.<name>.pwm-1` | float IN | PWM duty cycle channel 1 (Green) | 0.0 - 100.0 |
| `lcec.0.<name>.pwm-2` | float IN | PWM duty cycle channel 2 (Blue) | 0.0 - 100.0 |
| `lcec.0.<name>.pwm-3` | float IN | PWM duty cycle channel 3 (White) | 0.0 - 100.0 |
| `lcec.0.<name>.enable-0` | bit IN | Enable channel 0 | TRUE/FALSE |
| `lcec.0.<name>.enable-1` | bit IN | Enable channel 1 | TRUE/FALSE |
| `lcec.0.<name>.enable-2` | bit IN | Enable channel 2 | TRUE/FALSE |
| `lcec.0.<name>.enable-3` | bit IN | Enable channel 3 | TRUE/FALSE |

### Output Pins (Status)

| Pin | Type | Description |
|-----|------|-------------|
| `lcec.0.<name>.warning-0` | bit OUT | Warning channel 0 (e.g., open load) |
| `lcec.0.<name>.warning-1` | bit OUT | Warning channel 1 |
| `lcec.0.<name>.warning-2` | bit OUT | Warning channel 2 |
| `lcec.0.<name>.warning-3` | bit OUT | Warning channel 3 |
| `lcec.0.<name>.error-0` | bit OUT | Error channel 0 |
| `lcec.0.<name>.error-1` | bit OUT | Error channel 1 |
| `lcec.0.<name>.error-2` | bit OUT | Error channel 2 |
| `lcec.0.<name>.error-3` | bit OUT | Error channel 3 |
| `lcec.0.<name>.slave-online` | bit OUT | Slave is online |
| `lcec.0.<name>.slave-oper` | bit OUT | Slave is operational |
| `lcec.0.<name>.slave-state-op` | bit OUT | Slave in OP state |

---

## HAL Parameters Reference

### Per-Channel Parameters (Read-Write)

| Parameter | Type | Default | Description |
|-----------|------|---------|-------------|
| `lcec.0.<name>.scale-0` | float RW | 100.0 | Scale factor for PWM value |
| `lcec.0.<name>.scale-1` | float RW | 100.0 | Scale factor for PWM value |
| `lcec.0.<name>.scale-2` | float RW | 100.0 | Scale factor for PWM value |
| `lcec.0.<name>.scale-3` | float RW | 100.0 | Scale factor for PWM value |
| `lcec.0.<name>.offset-0` | float RW | 0.0 | Offset added to PWM value |
| `lcec.0.<name>.offset-1` | float RW | 0.0 | Offset added to PWM value |
| `lcec.0.<name>.offset-2` | float RW | 0.0 | Offset added to PWM value |
| `lcec.0.<name>.offset-3` | float RW | 0.0 | Offset added to PWM value |

**Calculation:** `Output = (pwm × scale + offset)`, clamped to 0-100%

### Per-Channel Parameters (Read-Only, from hardware)

| Parameter | Type | Description | SDO Address |
|-----------|------|-------------|-------------|
| `lcec.0.<name>.gamma-0..3` | float RO | Gamma correction (1.0 = linear) | 0x8000:0x24 |
| `lcec.0.<name>.ramp-time-0..3` | float RO | Ramp time in seconds | 0x8000:0x25 |

### Global Parameters (Read-Only)

| Parameter | Type | Default | Description | SDO Address |
|-----------|------|---------|-------------|-------------|
| `lcec.0.<name>.frequency` | u32 RO | 5000 | PWM frequency in Hz | 0xf819:0x11 |
| `lcec.0.<name>.master-gain` | s32 RO | 32767 | Master brightness (0-32767) | 0xf819:0x12 |

---

## Wiring Guide

### Common Anode RGB(W) LED Strip

```
                    +------------------+
                    |     EL2564       |
                    |                  |
PSU +24V  ─────────>| Terminal 3/7    |
                    | (LED supply+)    |
                    |                  |
PSU GND   ─────────>| Terminal 4/8    |
                    | (LED supply-)    |
                    |                  |
LED Strip           |                  |
  Anode (+) ───────>| Terminal 3/7    |
  Red (-)   <───────| Terminal 1      | (Output LED 1)
  Green (-) <───────| Terminal 5      | (Output LED 2)
  Blue (-)  <───────| Terminal 2      | (Output LED 3)
  White (-) <───────| Terminal 6      | (Output LED 4)
                    +------------------+
```

**IMPORTANT:**
- The EL2564 is a **Common Anode** driver (switches ground/cathode)
- LED power supply is **separate** from E-Bus power
- Supported voltage range: 5-48V DC
- Maximum current: 4A per channel

---

## Example Configurations

### Static Color Display

```hal
# Red
setp lcec.0.EL2564-1.pwm-0 100
setp lcec.0.EL2564-1.pwm-1 0
setp lcec.0.EL2564-1.pwm-2 0

# Green
setp lcec.0.EL2564-1.pwm-0 0
setp lcec.0.EL2564-1.pwm-1 100
setp lcec.0.EL2564-1.pwm-2 0

# Blue
setp lcec.0.EL2564-1.pwm-0 0
setp lcec.0.EL2564-1.pwm-1 0
setp lcec.0.EL2564-1.pwm-2 100

# Yellow (Red + Green)
setp lcec.0.EL2564-1.pwm-0 100
setp lcec.0.EL2564-1.pwm-1 100
setp lcec.0.EL2564-1.pwm-2 0

# Cyan (Green + Blue)
setp lcec.0.EL2564-1.pwm-0 0
setp lcec.0.EL2564-1.pwm-1 100
setp lcec.0.EL2564-1.pwm-2 100

# Magenta (Red + Blue)
setp lcec.0.EL2564-1.pwm-0 100
setp lcec.0.EL2564-1.pwm-1 0
setp lcec.0.EL2564-1.pwm-2 100

# White (RGB)
setp lcec.0.EL2564-1.pwm-0 100
setp lcec.0.EL2564-1.pwm-1 100
setp lcec.0.EL2564-1.pwm-2 100

# Orange
setp lcec.0.EL2564-1.pwm-0 100
setp lcec.0.EL2564-1.pwm-1 50
setp lcec.0.EL2564-1.pwm-2 0
```

### Machine State Indicator

```hal
# Connect LED to machine states

# Green when machine is enabled
net machine-on halui.machine.is-on => lcec.0.EL2564-1.enable-1
setp lcec.0.EL2564-1.pwm-1 100

# Red when estop is active
net estop-active halui.estop.is-activated => lcec.0.EL2564-1.enable-0
setp lcec.0.EL2564-1.pwm-0 100

# Blue when program is running
net program-running halui.program.is-running => lcec.0.EL2564-1.enable-2
setp lcec.0.EL2564-1.pwm-2 100
```

### Breathing Effect (Smooth Pulsing)

The breathing effect creates a smooth fade-in/fade-out animation, simulating a "breathing" pattern. This is achieved using 10 brightness steps for smooth transitions.

**Timing Parameters:**
- 10 brightness levels: 0%, 10%, 20%, 30%, 40%, 50%, 60%, 70%, 80%, 90%, 100%
- Step interval: 0.08 seconds (80ms per step)
- Peak hold: 0.15 seconds at 100% brightness
- Cycle pause: 0.2-0.3 seconds between cycles

**Example - Single Color Breathing (Red):**

```hal
# Initialize
setp lcec.0.EL2564-1.enable-0 1
setp lcec.0.EL2564-1.scale-0 1.0

# Fade in (0 -> 100%)
setp lcec.0.EL2564-1.pwm-0 0
loadusr -w sleep 0.08
setp lcec.0.EL2564-1.pwm-0 10
loadusr -w sleep 0.08
setp lcec.0.EL2564-1.pwm-0 20
loadusr -w sleep 0.08
setp lcec.0.EL2564-1.pwm-0 30
loadusr -w sleep 0.08
setp lcec.0.EL2564-1.pwm-0 40
loadusr -w sleep 0.08
setp lcec.0.EL2564-1.pwm-0 50
loadusr -w sleep 0.08
setp lcec.0.EL2564-1.pwm-0 60
loadusr -w sleep 0.08
setp lcec.0.EL2564-1.pwm-0 70
loadusr -w sleep 0.08
setp lcec.0.EL2564-1.pwm-0 80
loadusr -w sleep 0.08
setp lcec.0.EL2564-1.pwm-0 90
loadusr -w sleep 0.08
setp lcec.0.EL2564-1.pwm-0 100
loadusr -w sleep 0.15   # Hold at peak

# Fade out (100% -> 0)
setp lcec.0.EL2564-1.pwm-0 90
loadusr -w sleep 0.08
setp lcec.0.EL2564-1.pwm-0 80
loadusr -w sleep 0.08
setp lcec.0.EL2564-1.pwm-0 70
loadusr -w sleep 0.08
setp lcec.0.EL2564-1.pwm-0 60
loadusr -w sleep 0.08
setp lcec.0.EL2564-1.pwm-0 50
loadusr -w sleep 0.08
setp lcec.0.EL2564-1.pwm-0 40
loadusr -w sleep 0.08
setp lcec.0.EL2564-1.pwm-0 30
loadusr -w sleep 0.08
setp lcec.0.EL2564-1.pwm-0 20
loadusr -w sleep 0.08
setp lcec.0.EL2564-1.pwm-0 10
loadusr -w sleep 0.08
setp lcec.0.EL2564-1.pwm-0 0
loadusr -w sleep 0.2    # Pause before next cycle
```

**Example - Mixed Color Breathing (Cyan = Green + Blue):**

```hal
# Fade in both channels simultaneously
setp lcec.0.EL2564-1.pwm-1 10
setp lcec.0.EL2564-1.pwm-2 10
loadusr -w sleep 0.08
setp lcec.0.EL2564-1.pwm-1 20
setp lcec.0.EL2564-1.pwm-2 20
loadusr -w sleep 0.08
# ... continue to 100%
```

**Available Demo File:**

A complete breathing demo showing all colors (Red, Green, Blue, Cyan, Magenta, White) is included:

```bash
halrun -f el2564_breathing_demo.hal
```

---

### Blinking Effect with siggen

```hal
# Load signal generator
loadrt siggen

# Configure for 1 Hz square wave (blink)
setp siggen.0.frequency 1.0
setp siggen.0.amplitude 50
setp siggen.0.offset 50

# Add to thread
addf siggen.0.update servo-thread

# Connect to LED (Red blinking)
net blink-signal siggen.0.square => lcec.0.EL2564-1.pwm-0
setp lcec.0.EL2564-1.enable-0 1
```

### Color Selection with mux

```hal
# Load multiplexer (8 inputs)
loadrt mux_generic config="ff8,ff8,ff8,ff8"
addf mux-gen.00 servo-thread
addf mux-gen.01 servo-thread
addf mux-gen.02 servo-thread
addf mux-gen.03 servo-thread

# Define colors for Red channel
setp mux-gen.00.in-float-00 0    # Off
setp mux-gen.00.in-float-01 100  # Red
setp mux-gen.00.in-float-02 0    # Green
setp mux-gen.00.in-float-03 0    # Blue
setp mux-gen.00.in-float-04 100  # Yellow
setp mux-gen.00.in-float-05 0    # Cyan
setp mux-gen.00.in-float-06 100  # Magenta
setp mux-gen.00.in-float-07 100  # White

# Similar for Green (mux-gen.01), Blue (mux-gen.02), White (mux-gen.03)

# Connect selection signal
newsig color-select u32
net color-select => mux-gen.00.sel-int
net color-select => mux-gen.01.sel-int
net color-select => mux-gen.02.sel-int
net color-select => mux-gen.03.sel-int

# Connect outputs to EL2564
net pwm-r mux-gen.00.out-float => lcec.0.EL2564-1.pwm-0
net pwm-g mux-gen.01.out-float => lcec.0.EL2564-1.pwm-1
net pwm-b mux-gen.02.out-float => lcec.0.EL2564-1.pwm-2
net pwm-w mux-gen.03.out-float => lcec.0.EL2564-1.pwm-3
```

---

## Troubleshooting

### Slave stays in PREOP

**Symptom:** Run LED blinks slowly, slave doesn't reach OP state

**Solutions:**
1. Restart LinuxCNC
2. Check: `ethercat slaves -p <position>`
3. Review dmesg for error messages: `dmesg | grep -i ethercat`

### Warning bits always TRUE

**Symptom:** `warning-0` through `warning-3` are TRUE

**Cause:** "Open Load Detection" - no load connected

**Solution:**
- Normal when no LEDs are connected
- Disappears when LEDs are connected and powered
- Can be safely ignored if no LEDs are used

### LEDs not lighting

**Checklist:**
1. Is slave in OP state? `ethercat slaves -p <position>`
2. Is enable pin set? `halcmd show pin | grep enable`
3. Is PWM value set? `halcmd show pin | grep pwm`
4. Is LED power supply connected? Check terminals 3/4 or 7/8
5. Is LED wiring correct? (Common Anode!)

### LED flickering

**Solution:** Increase PWM frequency via SDO before starting LinuxCNC:

```bash
# Set frequency to 15 kHz (recommended)
ethercat download -p <position> 0xf819 0x11 -t uint32 15000
```

---

## C Driver API Reference

The standalone C driver (`el2564_driver.c` and `el2564_driver.h`) provides a high-level API for LED control outside of LinuxCNC.

### Initialization

```c
#include "el2564_driver.h"

el2564_driver_t driver;

// Initialize with optional hardware callback
el2564_init(&driver, my_pwm_write_callback);

// Cleanup
el2564_deinit(&driver);
```

### Color Control

```c
// Using predefined colors
el2564_set_color(&driver, EL2564_COLOR_RED);
el2564_set_color(&driver, EL2564_COLOR_GREEN);
el2564_set_color(&driver, EL2564_COLOR_BLUE);

// Using RGB values (0-255)
el2564_set_rgb(&driver, 255, 128, 0);  // Orange

// Using RGBW values (0-255)
el2564_set_rgbw(&driver, 255, 200, 150, 128);  // Warm white

// Using HSV (Hue 0-360, Saturation 0-100, Value 0-100)
el2564_color_t rainbow = el2564_hsv_to_rgb(180.0f, 100.0f, 100.0f);
el2564_set_color(&driver, rainbow);
```

### Brightness Control

```c
// Global master brightness (0-100%)
el2564_set_master_brightness(&driver, 50.0f);

// Per-channel brightness (0-100%)
el2564_set_channel_brightness(&driver, EL2564_CH_RED, 75.0f);
```

### Blink Mode

```c
// Blink a single channel (on_time_ms, off_time_ms)
el2564_blink_channel(&driver, EL2564_CH_RED, 500, 500);

// Blink a color
el2564_blink_color(&driver, EL2564_COLOR_YELLOW, 100, 100);

// Stop blinking
el2564_blink_stop(&driver, EL2564_CH_RED);
el2564_blink_stop_all(&driver);

// Must be called regularly to update blink states
el2564_update(&driver);
```

### Predefined Colors

```c
EL2564_COLOR_OFF        // All channels off
EL2564_COLOR_RED        // Pure red
EL2564_COLOR_GREEN      // Pure green
EL2564_COLOR_BLUE       // Pure blue
EL2564_COLOR_WHITE      // RGB white (no W channel)
EL2564_COLOR_YELLOW     // Red + Green
EL2564_COLOR_CYAN       // Green + Blue
EL2564_COLOR_MAGENTA    // Red + Blue
EL2564_COLOR_ORANGE     // Red + half Green
EL2564_COLOR_PURPLE     // Half Red + Blue
EL2564_COLOR_PINK       // Pink
EL2564_COLOR_WARM_WHITE // RGB + W for warm white
```

---

## SDO Configuration (Advanced)

To modify hardware parameters, use the `ethercat` command **before** starting LinuxCNC:

### PWM Frequency

```bash
# Set to 15 kHz (recommended for flicker-free operation)
ethercat download -p <pos> 0xf819 0x11 -t uint32 15000
```

### Master Gain

```bash
# Set to 50% (16383 of 32767)
ethercat download -p <pos> 0xf819 0x12 -t int16 16383
```

### Gamma Correction

```bash
# Set gamma 2.2 for channel 0 (recommended for LEDs)
ethercat download -p <pos> 0x8000 0x24 -t float 2.2
# Channel 1: 0x8010, Channel 2: 0x8020, Channel 3: 0x8030
```

### Ramp Time (Fade)

```bash
# Set 0.5 second fade for channel 0
ethercat download -p <pos> 0x8000 0x25 -t float 0.5
```

---

## License

Copyright (C) 2025 LinuxCNC EtherCAT

This program is free software under the GNU GPL v2 or later.

---

## Support

- LinuxCNC Forum: https://forum.linuxcnc.org/
- linuxcnc-ethercat: https://github.com/linuxcnc-ethercat/linuxcnc-ethercat
