/**
 * @file el2564_demo_main.c
 * @brief Hauptprogramm fuer EL2564 LED-Treiber Demo
 *
 * Kompilieren:
 *   gcc -o el2564_demo el2564_driver.c el2564_demo_main.c -lm
 *
 * Ausfuehren:
 *   ./el2564_demo          # Startet die vollstaendige Demo
 *   ./el2564_demo --help   # Zeigt Hilfe an
 */

/* POSIX Feature-Test-Macros (vor allen Includes!) */
#define _POSIX_C_SOURCE 199309L
#define _DEFAULT_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <sys/select.h>
#include <sys/time.h>
#include "el2564_driver.h"

/* Globaler Treiber fuer Signal-Handler */
static el2564_driver_t g_driver;
static volatile int g_running = 1;

/**
 * @brief Beispiel-Callback fuer PWM-Ausgabe
 *
 * In einer echten Anwendung wuerde hier die Hardware angesteuert,
 * z.B. ueber HAL-Pins oder direkte EtherCAT-Kommunikation.
 */
static void pwm_write_callback(int channel, uint16_t value) {
    const char *ch_names[] = {"R", "G", "B", "W"};
    float percent = (value / 65535.0f) * 100.0f;

    /* Ausgabe nur bei signifikanten Aenderungen (optional) */
    static uint16_t last_values[4] = {0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF};
    if (value != last_values[channel]) {
        printf("  [HW] Kanal %d (%s): PWM=%5d (%.1f%%)\n",
               channel, ch_names[channel], value, percent);
        last_values[channel] = value;
    }
}

/**
 * @brief Signal-Handler fuer sauberes Beenden
 */
static void signal_handler(int sig) {
    (void)sig;
    printf("\n[!] Signal empfangen, beende...\n");
    g_running = 0;
}

/**
 * @brief Zeigt die Hilfe an
 */
static void print_help(const char *prog_name) {
    printf("\nEL2564 LED-Treiber Demo\n");
    printf("=======================\n\n");
    printf("Verwendung: %s [OPTION]\n\n", prog_name);
    printf("Optionen:\n");
    printf("  --demo        Startet die vollstaendige Demo (Standard)\n");
    printf("  --colors      Zeigt alle vordefinierten Farben\n");
    printf("  --blink       Demonstriert den Blinkmodus\n");
    printf("  --rainbow     Zeigt einen Regenbogen-Effekt\n");
    printf("  --brightness  Demonstriert Helligkeitssteuerung\n");
    printf("  --interactive Interaktiver Modus\n");
    printf("  --help        Zeigt diese Hilfe an\n\n");
    printf("Beispiele:\n");
    printf("  %s              # Vollstaendige Demo\n", prog_name);
    printf("  %s --rainbow    # Nur Regenbogen-Effekt\n", prog_name);
    printf("  %s --blink      # Nur Blink-Demo\n\n", prog_name);
}

/**
 * @brief Zeigt alle vordefinierten Farben
 */
static void show_colors(el2564_driver_t *drv) {
    printf("\n=== Vordefinierte Farben ===\n\n");

    struct {
        const char *name;
        el2564_color_t color;
    } colors[] = {
        {"OFF (Aus)",        EL2564_COLOR_OFF},
        {"RED (Rot)",        EL2564_COLOR_RED},
        {"GREEN (Gruen)",    EL2564_COLOR_GREEN},
        {"BLUE (Blau)",      EL2564_COLOR_BLUE},
        {"WHITE (Weiss)",    EL2564_COLOR_WHITE},
        {"YELLOW (Gelb)",    EL2564_COLOR_YELLOW},
        {"CYAN (Tuerkis)",   EL2564_COLOR_CYAN},
        {"MAGENTA",          EL2564_COLOR_MAGENTA},
        {"ORANGE",           EL2564_COLOR_ORANGE},
        {"PURPLE (Lila)",    EL2564_COLOR_PURPLE},
        {"PINK (Rosa)",      EL2564_COLOR_PINK},
        {"WARM_WHITE",       EL2564_COLOR_WARM_WHITE},
    };

    for (size_t i = 0; i < sizeof(colors)/sizeof(colors[0]); i++) {
        printf("%-18s -> R:%3d  G:%3d  B:%3d  W:%3d\n",
               colors[i].name,
               colors[i].color.r,
               colors[i].color.g,
               colors[i].color.b,
               colors[i].color.w);

        el2564_set_color(drv, colors[i].color);
        usleep(800000);
    }

    el2564_all_off(drv);
    printf("\nFertig!\n");
}

/**
 * @brief Demonstriert den Blinkmodus
 */
static void demo_blink(el2564_driver_t *drv) {
    printf("\n=== Blink-Demonstration ===\n\n");

    /* Normales Blinken */
    printf("1. Rot blinkt normal (500ms/500ms)\n");
    el2564_set_color(drv, EL2564_COLOR_RED);
    el2564_blink_channel(drv, EL2564_CH_RED, 500, 500);

    for (int i = 0; i < 40 && g_running; i++) {
        el2564_update(drv);
        usleep(100000);
    }

    el2564_blink_stop_all(drv);

    /* Schnelles Blinken */
    printf("2. Gelb blinkt schnell (100ms/100ms)\n");
    el2564_set_color(drv, EL2564_COLOR_YELLOW);
    el2564_blink_color(drv, EL2564_COLOR_YELLOW, 100, 100);

    for (int i = 0; i < 40 && g_running; i++) {
        el2564_update(drv);
        usleep(100000);
    }

    el2564_blink_stop_all(drv);

    /* Asymmetrisches Blinken (kurz an, lang aus) */
    printf("3. Blau blinkt asymmetrisch (150ms an / 850ms aus)\n");
    el2564_set_color(drv, EL2564_COLOR_BLUE);
    el2564_blink_channel(drv, EL2564_CH_BLUE, 150, 850);

    for (int i = 0; i < 50 && g_running; i++) {
        el2564_update(drv);
        usleep(100000);
    }

    el2564_blink_stop_all(drv);

    /* SOS-Muster simulieren */
    printf("4. Weiss SOS-Muster (kurz-kurz-kurz, lang-lang-lang, kurz-kurz-kurz)\n");
    el2564_set_color(drv, EL2564_COLOR_WHITE);

    /* S: kurz kurz kurz */
    for (int i = 0; i < 3 && g_running; i++) {
        el2564_set_master_brightness(drv, 100.0f);
        usleep(200000);
        el2564_set_master_brightness(drv, 0.0f);
        usleep(200000);
    }
    usleep(400000);

    /* O: lang lang lang */
    for (int i = 0; i < 3 && g_running; i++) {
        el2564_set_master_brightness(drv, 100.0f);
        usleep(600000);
        el2564_set_master_brightness(drv, 0.0f);
        usleep(200000);
    }
    usleep(400000);

    /* S: kurz kurz kurz */
    for (int i = 0; i < 3 && g_running; i++) {
        el2564_set_master_brightness(drv, 100.0f);
        usleep(200000);
        el2564_set_master_brightness(drv, 0.0f);
        usleep(200000);
    }

    el2564_set_master_brightness(drv, 100.0f);
    el2564_all_off(drv);
    printf("\nFertig!\n");
}

/**
 * @brief Zeigt einen Regenbogen-Effekt
 */
static void demo_rainbow(el2564_driver_t *drv) {
    printf("\n=== Regenbogen-Effekt (HSV Farbkreis) ===\n");
    printf("Druecke Ctrl+C zum Beenden\n\n");

    float hue = 0;

    while (g_running) {
        el2564_color_t color = el2564_hsv_to_rgb(hue, 100.0f, 100.0f);
        el2564_set_color(drv, color);

        printf("\rHue: %5.1f°  RGB(%3d, %3d, %3d)  ",
               hue, color.r, color.g, color.b);
        fflush(stdout);

        hue += 1.0f;
        if (hue >= 360.0f) hue = 0.0f;

        usleep(30000);  /* ~33 FPS */
    }

    el2564_all_off(drv);
    printf("\n\nFertig!\n");
}

/**
 * @brief Demonstriert Helligkeitssteuerung
 */
static void demo_brightness(el2564_driver_t *drv) {
    printf("\n=== Helligkeits-Demonstration ===\n\n");

    /* Fade-In mit Weiss */
    printf("1. Fade-In (Weiss)\n");
    el2564_set_color(drv, EL2564_COLOR_WHITE);

    for (int b = 0; b <= 100 && g_running; b += 2) {
        el2564_set_master_brightness(drv, (float)b);
        printf("\r   Helligkeit: %3d%%", b);
        fflush(stdout);
        usleep(50000);
    }
    printf("\n");

    usleep(500000);

    /* Fade-Out */
    printf("2. Fade-Out\n");
    for (int b = 100; b >= 0 && g_running; b -= 2) {
        el2564_set_master_brightness(drv, (float)b);
        printf("\r   Helligkeit: %3d%%", b);
        fflush(stdout);
        usleep(50000);
    }
    printf("\n");

    el2564_set_master_brightness(drv, 100.0f);

    /* Individuelle Kanal-Helligkeit */
    printf("3. RGB-Kanaele nacheinander dimmen\n");

    const char *names[] = {"Rot", "Gruen", "Blau"};
    el2564_color_t colors[] = {EL2564_COLOR_RED, EL2564_COLOR_GREEN, EL2564_COLOR_BLUE};

    for (int ch = 0; ch < 3 && g_running; ch++) {
        el2564_set_color(drv, colors[ch]);
        printf("   %s:\n", names[ch]);

        for (int b = 0; b <= 100 && g_running; b += 5) {
            el2564_set_master_brightness(drv, (float)b);
            printf("\r      %3d%%", b);
            fflush(stdout);
            usleep(30000);
        }
        printf("\n");
        usleep(200000);
    }

    el2564_all_off(drv);
    el2564_set_master_brightness(drv, 100.0f);
    printf("\nFertig!\n");
}

/**
 * @brief Interaktiver Modus
 */
static void interactive_mode(el2564_driver_t *drv) {
    char input[256];
    int r, g, b, w;
    float brightness;

    printf("\n=== Interaktiver Modus ===\n");
    printf("Befehle:\n");
    printf("  rgb R G B        - Setze RGB-Farbe (0-255)\n");
    printf("  rgbw R G B W     - Setze RGBW-Farbe (0-255)\n");
    printf("  brightness N     - Setze Helligkeit (0-100)\n");
    printf("  blink CH ON OFF  - Blinken (Kanal 0-3, Zeiten in ms)\n");
    printf("  stop             - Blinken stoppen\n");
    printf("  off              - Alles aus\n");
    printf("  on               - Alles an\n");
    printf("  status           - Status anzeigen\n");
    printf("  quit             - Beenden\n\n");

    while (g_running) {
        printf("> ");
        fflush(stdout);

        if (!fgets(input, sizeof(input), stdin)) break;

        /* Newline entfernen */
        input[strcspn(input, "\n")] = 0;

        if (strlen(input) == 0) continue;

        if (strncmp(input, "rgb ", 4) == 0) {
            if (sscanf(input + 4, "%d %d %d", &r, &g, &b) == 3) {
                el2564_set_rgb(drv, (uint8_t)r, (uint8_t)g, (uint8_t)b);
                printf("Farbe gesetzt: R=%d G=%d B=%d\n", r, g, b);
            } else {
                printf("Fehler: rgb R G B (z.B. rgb 255 0 128)\n");
            }
        }
        else if (strncmp(input, "rgbw ", 5) == 0) {
            if (sscanf(input + 5, "%d %d %d %d", &r, &g, &b, &w) == 4) {
                el2564_set_rgbw(drv, (uint8_t)r, (uint8_t)g, (uint8_t)b, (uint8_t)w);
                printf("Farbe gesetzt: R=%d G=%d B=%d W=%d\n", r, g, b, w);
            } else {
                printf("Fehler: rgbw R G B W (z.B. rgbw 255 200 100 50)\n");
            }
        }
        else if (strncmp(input, "brightness ", 11) == 0) {
            if (sscanf(input + 11, "%f", &brightness) == 1) {
                el2564_set_master_brightness(drv, brightness);
                printf("Helligkeit: %.1f%%\n", brightness);
            } else {
                printf("Fehler: brightness N (z.B. brightness 50)\n");
            }
        }
        else if (strncmp(input, "blink ", 6) == 0) {
            int ch, on_ms, off_ms;
            if (sscanf(input + 6, "%d %d %d", &ch, &on_ms, &off_ms) == 3) {
                if (ch >= 0 && ch < 4) {
                    el2564_blink_channel(drv, ch, (uint32_t)on_ms, (uint32_t)off_ms);
                    printf("Kanal %d blinkt (%d ms an, %d ms aus)\n", ch, on_ms, off_ms);

                    /* Update-Schleife */
                    printf("Druecke Enter zum Stoppen...\n");
                    while (g_running) {
                        el2564_update(drv);
                        usleep(10000);

                        /* Non-blocking check fuer Enter */
                        fd_set fds;
                        struct timeval tv = {0, 0};
                        FD_ZERO(&fds);
                        FD_SET(0, &fds);
                        if (select(1, &fds, NULL, NULL, &tv) > 0) {
                            fgets(input, sizeof(input), stdin);
                            break;
                        }
                    }
                    el2564_blink_stop(drv, ch);
                } else {
                    printf("Fehler: Kanal muss 0-3 sein\n");
                }
            } else {
                printf("Fehler: blink CH ON OFF (z.B. blink 0 500 500)\n");
            }
        }
        else if (strcmp(input, "stop") == 0) {
            el2564_blink_stop_all(drv);
            printf("Blinken gestoppt\n");
        }
        else if (strcmp(input, "off") == 0) {
            el2564_all_off(drv);
            printf("Alle Kanaele aus\n");
        }
        else if (strcmp(input, "on") == 0) {
            el2564_all_on(drv);
            printf("Alle Kanaele an\n");
        }
        else if (strcmp(input, "status") == 0) {
            el2564_print_status(drv);
        }
        else if (strcmp(input, "quit") == 0 || strcmp(input, "exit") == 0) {
            break;
        }
        else {
            printf("Unbekannter Befehl: %s\n", input);
        }
    }

    el2564_all_off(drv);
}

/**
 * @brief Hauptprogramm
 */
int main(int argc, char *argv[]) {
    /* Signal-Handler einrichten */
    signal(SIGINT, signal_handler);
    signal(SIGTERM, signal_handler);

    /* Treiber initialisieren */
    if (el2564_init(&g_driver, pwm_write_callback) != 0) {
        fprintf(stderr, "Fehler: Treiber konnte nicht initialisiert werden!\n");
        return 1;
    }

    /* Argument parsen */
    if (argc > 1) {
        if (strcmp(argv[1], "--help") == 0 || strcmp(argv[1], "-h") == 0) {
            print_help(argv[0]);
        }
        else if (strcmp(argv[1], "--colors") == 0) {
            show_colors(&g_driver);
        }
        else if (strcmp(argv[1], "--blink") == 0) {
            demo_blink(&g_driver);
        }
        else if (strcmp(argv[1], "--rainbow") == 0) {
            demo_rainbow(&g_driver);
        }
        else if (strcmp(argv[1], "--brightness") == 0) {
            demo_brightness(&g_driver);
        }
        else if (strcmp(argv[1], "--interactive") == 0) {
            interactive_mode(&g_driver);
        }
        else if (strcmp(argv[1], "--demo") == 0) {
            el2564_demo(&g_driver);
        }
        else {
            fprintf(stderr, "Unbekannte Option: %s\n", argv[1]);
            print_help(argv[0]);
        }
    }
    else {
        /* Standard: vollstaendige Demo */
        el2564_demo(&g_driver);
    }

    /* Aufraeumen */
    el2564_deinit(&g_driver);

    return 0;
}
