/**
 * @file el2564_driver.c
 * @brief Beckhoff EL2564 4-Kanal PWM LED-Treiber Implementierung
 */

/* POSIX Feature-Test-Macros (vor allen Includes!) */
#define _POSIX_C_SOURCE 199309L
#define _DEFAULT_SOURCE

#include "el2564_driver.h"
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <sys/time.h>

#ifdef _WIN32
    #include <windows.h>
    #define SLEEP_MS(ms) Sleep(ms)
#else
    #include <unistd.h>
    #define SLEEP_MS(ms) usleep((ms) * 1000)
#endif

/* ============ Vordefinierte Farben ============ */

const el2564_color_t EL2564_COLOR_OFF        = {  0,   0,   0,   0};
const el2564_color_t EL2564_COLOR_RED        = {255,   0,   0,   0};
const el2564_color_t EL2564_COLOR_GREEN      = {  0, 255,   0,   0};
const el2564_color_t EL2564_COLOR_BLUE       = {  0,   0, 255,   0};
const el2564_color_t EL2564_COLOR_WHITE      = {255, 255, 255,   0};
const el2564_color_t EL2564_COLOR_YELLOW     = {255, 255,   0,   0};
const el2564_color_t EL2564_COLOR_CYAN       = {  0, 255, 255,   0};
const el2564_color_t EL2564_COLOR_MAGENTA    = {255,   0, 255,   0};
const el2564_color_t EL2564_COLOR_ORANGE     = {255, 128,   0,   0};
const el2564_color_t EL2564_COLOR_PURPLE     = {128,   0, 255,   0};
const el2564_color_t EL2564_COLOR_PINK       = {255, 105, 180,   0};
const el2564_color_t EL2564_COLOR_WARM_WHITE = {255, 200, 150, 128};

/* ============ Interne Hilfsfunktionen ============ */

static uint64_t get_time_ms(void) {
    struct timespec ts;
    clock_gettime(CLOCK_MONOTONIC, &ts);
    return (uint64_t)(ts.tv_sec * 1000 + ts.tv_nsec / 1000000);
}

static float clamp(float value, float min, float max) {
    if (value < min) return min;
    if (value > max) return max;
    return value;
}

static uint16_t percent_to_pwm(float percent) {
    percent = clamp(percent, 0.0f, 100.0f);
    return (uint16_t)((percent / 100.0f) * EL2564_PWM_MAX);
}

static void apply_output(el2564_driver_t *drv, int channel) {
    if (!drv || channel < 0 || channel >= EL2564_NUM_CHANNELS) return;

    el2564_channel_t *ch = &drv->channels[channel];
    uint16_t output = 0;

    if (ch->enabled && (!ch->blink.enabled || ch->blink.current_state)) {
        /* Berechne PWM mit Master-Helligkeit */
        float effective = (ch->brightness / 100.0f) * (drv->master_brightness / 100.0f);
        output = percent_to_pwm(effective * 100.0f);
    }

    ch->pwm_value = output;

    /* Callback aufrufen wenn vorhanden */
    if (drv->write_pwm) {
        drv->write_pwm(channel, output);
    }
}

static void apply_all_outputs(el2564_driver_t *drv) {
    for (int i = 0; i < EL2564_NUM_CHANNELS; i++) {
        apply_output(drv, i);
    }
}

/* ============ Initialisierung ============ */

int el2564_init(el2564_driver_t *drv, void (*write_callback)(int, uint16_t)) {
    if (!drv) return -1;

    memset(drv, 0, sizeof(el2564_driver_t));

    drv->master_brightness = 100.0f;
    drv->write_pwm = write_callback;

    /* Alle Kanaele initialisieren */
    for (int i = 0; i < EL2564_NUM_CHANNELS; i++) {
        drv->channels[i].brightness = 0.0f;
        drv->channels[i].pwm_value = 0;
        drv->channels[i].enabled = true;
        drv->channels[i].blink.enabled = false;
        drv->channels[i].blink.on_time_ms = 500;
        drv->channels[i].blink.off_time_ms = 500;
        drv->channels[i].blink.current_state = false;
        drv->channels[i].blink.last_toggle = 0;
    }

    drv->initialized = true;

    printf("[EL2564] Treiber initialisiert\n");
    return 0;
}

void el2564_deinit(el2564_driver_t *drv) {
    if (!drv) return;

    el2564_all_off(drv);
    drv->initialized = false;

    printf("[EL2564] Treiber deinitialisiert\n");
}

/* ============ Helligkeitssteuerung ============ */

void el2564_set_master_brightness(el2564_driver_t *drv, float brightness) {
    if (!drv) return;

    drv->master_brightness = clamp(brightness, 0.0f, 100.0f);
    apply_all_outputs(drv);
}

float el2564_get_master_brightness(el2564_driver_t *drv) {
    return drv ? drv->master_brightness : 0.0f;
}

void el2564_set_channel_brightness(el2564_driver_t *drv, int channel, float brightness) {
    if (!drv || channel < 0 || channel >= EL2564_NUM_CHANNELS) return;

    drv->channels[channel].brightness = clamp(brightness, 0.0f, 100.0f);
    apply_output(drv, channel);
}

/* ============ Farbsteuerung ============ */

el2564_color_t el2564_make_color(uint8_t r, uint8_t g, uint8_t b, uint8_t w) {
    el2564_color_t color = {r, g, b, w};
    return color;
}

el2564_color_t el2564_hsv_to_rgb(float h, float s, float v) {
    el2564_color_t color = {0, 0, 0, 0};

    s = clamp(s, 0.0f, 100.0f) / 100.0f;
    v = clamp(v, 0.0f, 100.0f) / 100.0f;

    /* Normalisiere Hue auf 0-360 */
    while (h < 0) h += 360;
    while (h >= 360) h -= 360;

    float c = v * s;
    float x = c * (1 - fabsf(fmodf(h / 60.0f, 2) - 1));
    float m = v - c;

    float r, g, b;

    if (h < 60) {
        r = c; g = x; b = 0;
    } else if (h < 120) {
        r = x; g = c; b = 0;
    } else if (h < 180) {
        r = 0; g = c; b = x;
    } else if (h < 240) {
        r = 0; g = x; b = c;
    } else if (h < 300) {
        r = x; g = 0; b = c;
    } else {
        r = c; g = 0; b = x;
    }

    color.r = (uint8_t)((r + m) * 255);
    color.g = (uint8_t)((g + m) * 255);
    color.b = (uint8_t)((b + m) * 255);

    return color;
}

void el2564_set_color(el2564_driver_t *drv, el2564_color_t color) {
    if (!drv) return;

    drv->current_color = color;

    /* Konvertiere 8-bit Farbwerte zu Prozent-Helligkeit */
    drv->channels[EL2564_CH_RED].brightness = (color.r / 255.0f) * 100.0f;
    drv->channels[EL2564_CH_GREEN].brightness = (color.g / 255.0f) * 100.0f;
    drv->channels[EL2564_CH_BLUE].brightness = (color.b / 255.0f) * 100.0f;
    drv->channels[EL2564_CH_WHITE].brightness = (color.w / 255.0f) * 100.0f;

    apply_all_outputs(drv);
}

void el2564_set_rgbw(el2564_driver_t *drv, uint8_t r, uint8_t g, uint8_t b, uint8_t w) {
    el2564_set_color(drv, el2564_make_color(r, g, b, w));
}

void el2564_set_rgb(el2564_driver_t *drv, uint8_t r, uint8_t g, uint8_t b) {
    if (!drv) return;

    drv->current_color.r = r;
    drv->current_color.g = g;
    drv->current_color.b = b;

    drv->channels[EL2564_CH_RED].brightness = (r / 255.0f) * 100.0f;
    drv->channels[EL2564_CH_GREEN].brightness = (g / 255.0f) * 100.0f;
    drv->channels[EL2564_CH_BLUE].brightness = (b / 255.0f) * 100.0f;

    apply_output(drv, EL2564_CH_RED);
    apply_output(drv, EL2564_CH_GREEN);
    apply_output(drv, EL2564_CH_BLUE);
}

/* ============ Kanalsteuerung ============ */

void el2564_set_channel_enabled(el2564_driver_t *drv, int channel, bool enabled) {
    if (!drv || channel < 0 || channel >= EL2564_NUM_CHANNELS) return;

    drv->channels[channel].enabled = enabled;
    apply_output(drv, channel);
}

void el2564_set_channel_pwm(el2564_driver_t *drv, int channel, uint16_t pwm_value) {
    if (!drv || channel < 0 || channel >= EL2564_NUM_CHANNELS) return;

    drv->channels[channel].brightness = (pwm_value / (float)EL2564_PWM_MAX) * 100.0f;
    apply_output(drv, channel);
}

void el2564_all_off(el2564_driver_t *drv) {
    if (!drv) return;

    for (int i = 0; i < EL2564_NUM_CHANNELS; i++) {
        drv->channels[i].brightness = 0.0f;
        drv->channels[i].blink.enabled = false;
    }
    drv->current_color = EL2564_COLOR_OFF;
    apply_all_outputs(drv);
}

void el2564_all_on(el2564_driver_t *drv) {
    if (!drv) return;

    for (int i = 0; i < EL2564_NUM_CHANNELS; i++) {
        drv->channels[i].brightness = 100.0f;
    }
    drv->current_color = EL2564_COLOR_WHITE;
    apply_all_outputs(drv);
}

/* ============ Blinkmodus ============ */

void el2564_blink_channel(el2564_driver_t *drv, int channel,
                          uint32_t on_time_ms, uint32_t off_time_ms) {
    if (!drv || channel < 0 || channel >= EL2564_NUM_CHANNELS) return;

    el2564_blink_t *blink = &drv->channels[channel].blink;
    blink->enabled = true;
    blink->on_time_ms = on_time_ms;
    blink->off_time_ms = off_time_ms;
    blink->current_state = true;
    blink->last_toggle = get_time_ms();

    apply_output(drv, channel);
}

void el2564_blink_color(el2564_driver_t *drv, el2564_color_t color,
                        uint32_t on_time_ms, uint32_t off_time_ms) {
    if (!drv) return;

    /* Setze die Farbe */
    el2564_set_color(drv, color);

    /* Aktiviere Blinken fuer alle Kanaele mit Farbwert > 0 */
    if (color.r > 0) el2564_blink_channel(drv, EL2564_CH_RED, on_time_ms, off_time_ms);
    if (color.g > 0) el2564_blink_channel(drv, EL2564_CH_GREEN, on_time_ms, off_time_ms);
    if (color.b > 0) el2564_blink_channel(drv, EL2564_CH_BLUE, on_time_ms, off_time_ms);
    if (color.w > 0) el2564_blink_channel(drv, EL2564_CH_WHITE, on_time_ms, off_time_ms);
}

void el2564_blink_stop(el2564_driver_t *drv, int channel) {
    if (!drv || channel < 0 || channel >= EL2564_NUM_CHANNELS) return;

    drv->channels[channel].blink.enabled = false;
    drv->channels[channel].blink.current_state = false;
    apply_output(drv, channel);
}

void el2564_blink_stop_all(el2564_driver_t *drv) {
    if (!drv) return;

    for (int i = 0; i < EL2564_NUM_CHANNELS; i++) {
        drv->channels[i].blink.enabled = false;
        drv->channels[i].blink.current_state = false;
    }
    apply_all_outputs(drv);
}

void el2564_update(el2564_driver_t *drv) {
    if (!drv) return;

    uint64_t now = get_time_ms();

    for (int i = 0; i < EL2564_NUM_CHANNELS; i++) {
        el2564_blink_t *blink = &drv->channels[i].blink;

        if (!blink->enabled) continue;

        uint32_t interval = blink->current_state ? blink->on_time_ms : blink->off_time_ms;

        if (now - blink->last_toggle >= interval) {
            blink->current_state = !blink->current_state;
            blink->last_toggle = now;
            apply_output(drv, i);
        }
    }
}

/* ============ Hilfsfunktionen ============ */

const char* el2564_color_name(el2564_color_t color) {
    if (color.r == 0 && color.g == 0 && color.b == 0 && color.w == 0) return "Aus";
    if (color.r == 255 && color.g == 0 && color.b == 0) return "Rot";
    if (color.r == 0 && color.g == 255 && color.b == 0) return "Gruen";
    if (color.r == 0 && color.g == 0 && color.b == 255) return "Blau";
    if (color.r == 255 && color.g == 255 && color.b == 255) return "Weiss";
    if (color.r == 255 && color.g == 255 && color.b == 0) return "Gelb";
    if (color.r == 0 && color.g == 255 && color.b == 255) return "Cyan";
    if (color.r == 255 && color.g == 0 && color.b == 255) return "Magenta";
    if (color.r == 255 && color.g == 128 && color.b == 0) return "Orange";
    if (color.r == 128 && color.g == 0 && color.b == 255) return "Lila";
    return "Benutzerdefiniert";
}

void el2564_print_status(el2564_driver_t *drv) {
    if (!drv) {
        printf("[EL2564] Treiber nicht initialisiert!\n");
        return;
    }

    printf("\n===== EL2564 Status =====\n");
    printf("Master-Helligkeit: %.1f%%\n", drv->master_brightness);
    printf("Aktuelle Farbe: %s (R:%d G:%d B:%d W:%d)\n",
           el2564_color_name(drv->current_color),
           drv->current_color.r, drv->current_color.g,
           drv->current_color.b, drv->current_color.w);
    printf("\nKanaele:\n");

    const char* ch_names[] = {"Rot  ", "Gruen", "Blau ", "Weiss"};

    for (int i = 0; i < EL2564_NUM_CHANNELS; i++) {
        el2564_channel_t *ch = &drv->channels[i];
        printf("  [%d] %s: %6.1f%% | PWM: %5d | %s | Blink: %s\n",
               i, ch_names[i],
               ch->brightness,
               ch->pwm_value,
               ch->enabled ? "EIN " : "AUS ",
               ch->blink.enabled ? "AN" : "AUS");
    }
    printf("=========================\n\n");
}

/* ============ Demo-Funktion ============ */

static void demo_print(const char *text) {
    printf("\n>>> %s\n", text);
}

void el2564_demo(el2564_driver_t *drv) {
    if (!drv) {
        printf("[EL2564] Fehler: Treiber nicht initialisiert!\n");
        return;
    }

    printf("\n");
    printf("╔═══════════════════════════════════════════════════════════╗\n");
    printf("║        EL2564 LED-Treiber Demonstration                   ║\n");
    printf("║        Beckhoff 4-Kanal PWM RGBW Controller               ║\n");
    printf("╚═══════════════════════════════════════════════════════════╝\n");

    /* 1. Grundfarben */
    demo_print("1. Grundfarben anzeigen");

    struct {
        const char *name;
        el2564_color_t color;
    } colors[] = {
        {"Rot",      EL2564_COLOR_RED},
        {"Gruen",    EL2564_COLOR_GREEN},
        {"Blau",     EL2564_COLOR_BLUE},
        {"Gelb",     EL2564_COLOR_YELLOW},
        {"Cyan",     EL2564_COLOR_CYAN},
        {"Magenta",  EL2564_COLOR_MAGENTA},
        {"Orange",   EL2564_COLOR_ORANGE},
        {"Lila",     EL2564_COLOR_PURPLE},
        {"Pink",     EL2564_COLOR_PINK},
        {"Weiss",    EL2564_COLOR_WHITE},
    };

    for (size_t i = 0; i < sizeof(colors)/sizeof(colors[0]); i++) {
        printf("   -> %s\n", colors[i].name);
        el2564_set_color(drv, colors[i].color);
        el2564_print_status(drv);
        SLEEP_MS(800);
    }

    /* 2. Helligkeitsregelung */
    demo_print("2. Helligkeitsregelung (Rot bei verschiedenen Helligkeiten)");

    el2564_set_color(drv, EL2564_COLOR_RED);

    float brightness_levels[] = {100.0f, 75.0f, 50.0f, 25.0f, 10.0f, 5.0f, 100.0f};
    for (size_t i = 0; i < sizeof(brightness_levels)/sizeof(brightness_levels[0]); i++) {
        printf("   -> Master-Helligkeit: %.0f%%\n", brightness_levels[i]);
        el2564_set_master_brightness(drv, brightness_levels[i]);
        el2564_print_status(drv);
        SLEEP_MS(600);
    }

    /* 3. Einzelkanal-Helligkeit */
    demo_print("3. Einzelkanal-Helligkeit (individuelle RGB-Steuerung)");

    el2564_all_off(drv);
    el2564_set_master_brightness(drv, 100.0f);

    printf("   -> Rot hochdimmen\n");
    for (int b = 0; b <= 100; b += 20) {
        el2564_set_channel_brightness(drv, EL2564_CH_RED, (float)b);
        printf("      R=%d%%\n", b);
        SLEEP_MS(200);
    }

    printf("   -> Gruen hochdimmen\n");
    for (int b = 0; b <= 100; b += 20) {
        el2564_set_channel_brightness(drv, EL2564_CH_GREEN, (float)b);
        printf("      G=%d%%\n", b);
        SLEEP_MS(200);
    }

    printf("   -> Blau hochdimmen\n");
    for (int b = 0; b <= 100; b += 20) {
        el2564_set_channel_brightness(drv, EL2564_CH_BLUE, (float)b);
        printf("      B=%d%%\n", b);
        SLEEP_MS(200);
    }

    el2564_print_status(drv);
    SLEEP_MS(500);

    /* 4. HSV Farbkreis */
    demo_print("4. HSV Farbkreis (Regenbogen-Effekt)");

    for (int h = 0; h < 360; h += 15) {
        el2564_color_t rainbow = el2564_hsv_to_rgb((float)h, 100.0f, 100.0f);
        el2564_set_color(drv, rainbow);
        printf("   -> Hue: %3d° (R:%3d G:%3d B:%3d)\n", h, rainbow.r, rainbow.g, rainbow.b);
        SLEEP_MS(150);
    }

    /* 5. Blinkmodus */
    demo_print("5. Blinkmodus - Rot blinkt (500ms an/500ms aus)");

    el2564_set_color(drv, EL2564_COLOR_RED);
    el2564_blink_channel(drv, EL2564_CH_RED, 500, 500);

    printf("   -> Blinke 5 Sekunden...\n");
    for (int i = 0; i < 50; i++) {
        el2564_update(drv);
        SLEEP_MS(100);
    }

    el2564_blink_stop_all(drv);
    el2564_print_status(drv);

    /* 6. Schnelles Blinken */
    demo_print("6. Schnelles Blinken - Gelb (100ms an/100ms aus)");

    el2564_set_color(drv, EL2564_COLOR_YELLOW);
    el2564_blink_color(drv, EL2564_COLOR_YELLOW, 100, 100);

    printf("   -> Schnelles Blinken 3 Sekunden...\n");
    for (int i = 0; i < 60; i++) {
        el2564_update(drv);
        SLEEP_MS(50);
    }

    el2564_blink_stop_all(drv);

    /* 7. Asymmetrisches Blinken */
    demo_print("7. Asymmetrisches Blinken - Blau (200ms an/800ms aus)");

    el2564_set_color(drv, EL2564_COLOR_BLUE);
    el2564_blink_channel(drv, EL2564_CH_BLUE, 200, 800);

    printf("   -> Asymmetrisches Blinken 5 Sekunden...\n");
    for (int i = 0; i < 100; i++) {
        el2564_update(drv);
        SLEEP_MS(50);
    }

    el2564_blink_stop_all(drv);

    /* 8. RGBW Warmweiss */
    demo_print("8. RGBW Warmweiss (mit Weiss-Kanal)");

    el2564_set_color(drv, EL2564_COLOR_WARM_WHITE);
    el2564_print_status(drv);
    SLEEP_MS(1500);

    /* 9. Benutzerdefinierte Farbe */
    demo_print("9. Benutzerdefinierte Farben");

    printf("   -> Koralle (255, 127, 80)\n");
    el2564_set_rgb(drv, 255, 127, 80);
    el2564_print_status(drv);
    SLEEP_MS(1000);

    printf("   -> Tuerkis (64, 224, 208)\n");
    el2564_set_rgb(drv, 64, 224, 208);
    el2564_print_status(drv);
    SLEEP_MS(1000);

    printf("   -> Gold (255, 215, 0)\n");
    el2564_set_rgb(drv, 255, 215, 0);
    el2564_print_status(drv);
    SLEEP_MS(1000);

    /* 10. Fade-Out */
    demo_print("10. Fade-Out Effekt");

    el2564_set_color(drv, EL2564_COLOR_WHITE);
    for (int b = 100; b >= 0; b -= 5) {
        el2564_set_master_brightness(drv, (float)b);
        printf("   -> Helligkeit: %d%%\n", b);
        SLEEP_MS(100);
    }

    /* Ende */
    el2564_all_off(drv);
    el2564_set_master_brightness(drv, 100.0f);

    printf("\n");
    printf("╔═══════════════════════════════════════════════════════════╗\n");
    printf("║              Demo beendet!                                ║\n");
    printf("╚═══════════════════════════════════════════════════════════╝\n\n");
}
