/**
 * @file el2564_driver.h
 * @brief Beckhoff EL2564 4-Kanal PWM LED-Treiber
 *
 * Treiber fuer RGBW LED-Steuerung mit:
 * - Helligkeitsregelung (0-100%)
 * - Vordefinierte und benutzerdefinierte Farben
 * - Blinkmodus pro Kanal/Farbe
 */

#ifndef EL2564_DRIVER_H
#define EL2564_DRIVER_H

#include <stdint.h>
#include <stdbool.h>

#ifdef __cplusplus
extern "C" {
#endif

/* Anzahl der PWM-Kanaele */
#define EL2564_NUM_CHANNELS 4

/* Kanal-Indizes */
#define EL2564_CH_RED    0
#define EL2564_CH_GREEN  1
#define EL2564_CH_BLUE   2
#define EL2564_CH_WHITE  3

/* PWM Wertebereiche */
#define EL2564_PWM_MIN   0
#define EL2564_PWM_MAX   65535
#define EL2564_PERCENT_MAX 100.0f

/**
 * @brief RGBW Farbstruktur
 */
typedef struct {
    uint8_t r;  /**< Rot (0-255) */
    uint8_t g;  /**< Gruen (0-255) */
    uint8_t b;  /**< Blau (0-255) */
    uint8_t w;  /**< Weiss (0-255) */
} el2564_color_t;

/**
 * @brief Blinkmodus-Konfiguration
 */
typedef struct {
    bool enabled;           /**< Blinken aktiv */
    uint32_t on_time_ms;    /**< Ein-Zeit in Millisekunden */
    uint32_t off_time_ms;   /**< Aus-Zeit in Millisekunden */
    bool current_state;     /**< Aktueller Zustand (intern) */
    uint64_t last_toggle;   /**< Letzter Umschaltpunkt (intern) */
} el2564_blink_t;

/**
 * @brief Einzelner PWM-Kanal
 */
typedef struct {
    uint16_t pwm_value;     /**< Aktueller PWM-Wert (0-65535) */
    float brightness;       /**< Helligkeit in Prozent (0-100) */
    bool enabled;           /**< Kanal aktiviert */
    el2564_blink_t blink;   /**< Blink-Konfiguration */
} el2564_channel_t;

/**
 * @brief Haupt-Treiberstruktur
 */
typedef struct {
    el2564_channel_t channels[EL2564_NUM_CHANNELS];
    float master_brightness;  /**< Globale Helligkeit (0-100%) */
    el2564_color_t current_color;
    bool initialized;

    /* Callback fuer Hardware-Ausgabe (muss implementiert werden) */
    void (*write_pwm)(int channel, uint16_t value);
} el2564_driver_t;

/* ============ Vordefinierte Farben ============ */

extern const el2564_color_t EL2564_COLOR_OFF;
extern const el2564_color_t EL2564_COLOR_RED;
extern const el2564_color_t EL2564_COLOR_GREEN;
extern const el2564_color_t EL2564_COLOR_BLUE;
extern const el2564_color_t EL2564_COLOR_WHITE;
extern const el2564_color_t EL2564_COLOR_YELLOW;
extern const el2564_color_t EL2564_COLOR_CYAN;
extern const el2564_color_t EL2564_COLOR_MAGENTA;
extern const el2564_color_t EL2564_COLOR_ORANGE;
extern const el2564_color_t EL2564_COLOR_PURPLE;
extern const el2564_color_t EL2564_COLOR_PINK;
extern const el2564_color_t EL2564_COLOR_WARM_WHITE;

/* ============ Initialisierung ============ */

/**
 * @brief Initialisiert den EL2564 Treiber
 * @param drv Zeiger auf Treiberstruktur
 * @param write_callback Callback-Funktion fuer PWM-Ausgabe (kann NULL sein fuer Demo)
 * @return 0 bei Erfolg, -1 bei Fehler
 */
int el2564_init(el2564_driver_t *drv, void (*write_callback)(int, uint16_t));

/**
 * @brief Deinitialisiert den Treiber
 * @param drv Zeiger auf Treiberstruktur
 */
void el2564_deinit(el2564_driver_t *drv);

/* ============ Helligkeitssteuerung ============ */

/**
 * @brief Setzt die globale Master-Helligkeit
 * @param drv Zeiger auf Treiberstruktur
 * @param brightness Helligkeit in Prozent (0.0 - 100.0)
 */
void el2564_set_master_brightness(el2564_driver_t *drv, float brightness);

/**
 * @brief Gibt die aktuelle Master-Helligkeit zurueck
 * @param drv Zeiger auf Treiberstruktur
 * @return Helligkeit in Prozent
 */
float el2564_get_master_brightness(el2564_driver_t *drv);

/**
 * @brief Setzt die Helligkeit eines einzelnen Kanals
 * @param drv Zeiger auf Treiberstruktur
 * @param channel Kanal-Index (0-3)
 * @param brightness Helligkeit in Prozent (0.0 - 100.0)
 */
void el2564_set_channel_brightness(el2564_driver_t *drv, int channel, float brightness);

/* ============ Farbsteuerung ============ */

/**
 * @brief Setzt eine RGBW-Farbe
 * @param drv Zeiger auf Treiberstruktur
 * @param color Farbstruktur
 */
void el2564_set_color(el2564_driver_t *drv, el2564_color_t color);

/**
 * @brief Setzt eine RGBW-Farbe mit separaten Werten
 * @param drv Zeiger auf Treiberstruktur
 * @param r Rot (0-255)
 * @param g Gruen (0-255)
 * @param b Blau (0-255)
 * @param w Weiss (0-255)
 */
void el2564_set_rgbw(el2564_driver_t *drv, uint8_t r, uint8_t g, uint8_t b, uint8_t w);

/**
 * @brief Setzt eine RGB-Farbe (Weiss-Kanal bleibt unveraendert)
 * @param drv Zeiger auf Treiberstruktur
 * @param r Rot (0-255)
 * @param g Gruen (0-255)
 * @param b Blau (0-255)
 */
void el2564_set_rgb(el2564_driver_t *drv, uint8_t r, uint8_t g, uint8_t b);

/**
 * @brief Erstellt eine Farbe aus RGB-Werten
 * @return Farbstruktur
 */
el2564_color_t el2564_make_color(uint8_t r, uint8_t g, uint8_t b, uint8_t w);

/**
 * @brief Erstellt eine Farbe aus HSV-Werten
 * @param h Farbton (0-360)
 * @param s Saettigung (0-100)
 * @param v Helligkeit (0-100)
 * @return RGB Farbstruktur (W=0)
 */
el2564_color_t el2564_hsv_to_rgb(float h, float s, float v);

/* ============ Kanalsteuerung ============ */

/**
 * @brief Aktiviert/Deaktiviert einen Kanal
 * @param drv Zeiger auf Treiberstruktur
 * @param channel Kanal-Index (0-3)
 * @param enabled true = aktiviert, false = deaktiviert
 */
void el2564_set_channel_enabled(el2564_driver_t *drv, int channel, bool enabled);

/**
 * @brief Setzt den PWM-Wert eines Kanals direkt
 * @param drv Zeiger auf Treiberstruktur
 * @param channel Kanal-Index (0-3)
 * @param pwm_value PWM-Wert (0-65535)
 */
void el2564_set_channel_pwm(el2564_driver_t *drv, int channel, uint16_t pwm_value);

/**
 * @brief Schaltet alle Kanaele aus
 * @param drv Zeiger auf Treiberstruktur
 */
void el2564_all_off(el2564_driver_t *drv);

/**
 * @brief Schaltet alle Kanaele auf Maximum
 * @param drv Zeiger auf Treiberstruktur
 */
void el2564_all_on(el2564_driver_t *drv);

/* ============ Blinkmodus ============ */

/**
 * @brief Aktiviert den Blinkmodus fuer einen Kanal
 * @param drv Zeiger auf Treiberstruktur
 * @param channel Kanal-Index (0-3)
 * @param on_time_ms Ein-Zeit in Millisekunden
 * @param off_time_ms Aus-Zeit in Millisekunden
 */
void el2564_blink_channel(el2564_driver_t *drv, int channel,
                          uint32_t on_time_ms, uint32_t off_time_ms);

/**
 * @brief Aktiviert den Blinkmodus fuer eine Farbe
 * @param drv Zeiger auf Treiberstruktur
 * @param color Farbe die blinken soll
 * @param on_time_ms Ein-Zeit in Millisekunden
 * @param off_time_ms Aus-Zeit in Millisekunden
 */
void el2564_blink_color(el2564_driver_t *drv, el2564_color_t color,
                        uint32_t on_time_ms, uint32_t off_time_ms);

/**
 * @brief Deaktiviert den Blinkmodus fuer einen Kanal
 * @param drv Zeiger auf Treiberstruktur
 * @param channel Kanal-Index (0-3)
 */
void el2564_blink_stop(el2564_driver_t *drv, int channel);

/**
 * @brief Deaktiviert den Blinkmodus fuer alle Kanaele
 * @param drv Zeiger auf Treiberstruktur
 */
void el2564_blink_stop_all(el2564_driver_t *drv);

/**
 * @brief Aktualisiert die Blink-Zustaende (muss regelmaessig aufgerufen werden)
 * @param drv Zeiger auf Treiberstruktur
 */
void el2564_update(el2564_driver_t *drv);

/* ============ Hilfsfunktionen ============ */

/**
 * @brief Gibt den Namen einer vordefinierten Farbe zurueck
 * @param color Farbstruktur
 * @return Farbname als String
 */
const char* el2564_color_name(el2564_color_t color);

/**
 * @brief Druckt den aktuellen Treiberzustand
 * @param drv Zeiger auf Treiberstruktur
 */
void el2564_print_status(el2564_driver_t *drv);

/* ============ Demo-Funktion ============ */

/**
 * @brief Startet eine Demo, die alle Funktionen zeigt
 * @param drv Zeiger auf Treiberstruktur
 */
void el2564_demo(el2564_driver_t *drv);

#ifdef __cplusplus
}
#endif

#endif /* EL2564_DRIVER_H */
