#!/bin/bash
#
# LinuxCNC DB Tool - Debian Package Build Script
# Builds the complete package from source with all files included
#

set -e  # Exit on error

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Package information
PKG_NAME="linuxcnc-db-tool"
PKG_VERSION="1.0.0"
PKG_ARCH="amd64"
PKG_MAINTAINER="LinuxCNC Community <community@linuxcnc.org>"

# Directories
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
BUILD_DIR="${SCRIPT_DIR}/build-tmp"
PKG_DIR="${BUILD_DIR}/${PKG_NAME}_${PKG_VERSION}_${PKG_ARCH}"
SRC_DIR="${SCRIPT_DIR}/linuxcnc-db-tool-build"
OUTPUT_DIR="${SCRIPT_DIR}"

echo -e "${BLUE}╔══════════════════════════════════════════════════════════════════════════╗${NC}"
echo -e "${BLUE}║         LinuxCNC DB Tool - Debian Package Build Script                  ║${NC}"
echo -e "${BLUE}║                     Version ${PKG_VERSION}                                       ║${NC}"
echo -e "${BLUE}╚══════════════════════════════════════════════════════════════════════════╝${NC}"
echo

# Function to print status messages
print_status() {
    echo -e "${GREEN}[✓]${NC} $1"
}

print_info() {
    echo -e "${BLUE}[i]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[!]${NC} $1"
}

print_error() {
    echo -e "${RED}[✗]${NC} $1"
}

# Check for required tools
check_dependencies() {
    print_info "Checking build dependencies..."

    local missing_deps=()

    command -v gcc >/dev/null 2>&1 || missing_deps+=("gcc")
    command -v make >/dev/null 2>&1 || missing_deps+=("make")
    command -v dpkg-deb >/dev/null 2>&1 || missing_deps+=("dpkg-dev")
    command -v pkg-config >/dev/null 2>&1 || missing_deps+=("pkg-config")

    if [ ${#missing_deps[@]} -ne 0 ]; then
        print_error "Missing dependencies: ${missing_deps[*]}"
        echo "Install with: sudo apt-get install ${missing_deps[*]}"
        exit 1
    fi

    # Check GTK3
    if ! pkg-config --exists gtk+-3.0; then
        print_warning "GTK3 development files not found. GUI will not be built."
        print_info "Install with: sudo apt-get install libgtk-3-dev"
    fi

    print_status "All required dependencies found"
}

# Clean previous build
clean_build() {
    print_info "Cleaning previous build artifacts..."
    rm -rf "${BUILD_DIR}"
    rm -f "${OUTPUT_DIR}/${PKG_NAME}_${PKG_VERSION}_${PKG_ARCH}.deb"
    print_status "Clean complete"
}

# Create directory structure
create_structure() {
    print_info "Creating package directory structure..."

    mkdir -p "${PKG_DIR}/DEBIAN"
    mkdir -p "${PKG_DIR}/usr/local/bin"
    mkdir -p "${PKG_DIR}/usr/local/share/applications"
    mkdir -p "${PKG_DIR}/usr/local/share/doc/${PKG_NAME}"
    mkdir -p "${PKG_DIR}/usr/local/share/${PKG_NAME}/src"
    mkdir -p "${PKG_DIR}/usr/local/share/${PKG_NAME}/examples"

    print_status "Directory structure created"
}

# Compile source code
compile_binaries() {
    print_info "Compiling source code..."

    if [ ! -f "${SRC_DIR}/db_tool.c" ]; then
        print_error "Source files not found in ${SRC_DIR}/"
        print_info "Please ensure db_tool.c and db_tool_monitor.c are in the src/ directory"
        exit 1
    fi

    cd "${SRC_DIR}"

    # Compile db_tool
    print_info "  Compiling db_tool..."
    gcc -Wall -Wextra -O2 -pthread -o db_tool db_tool.c -pthread -lm

    # Compile db_tool_monitor and db_tool_manager (both from db_tool_manager.c)
    if pkg-config --exists gtk+-3.0; then
        print_info "  Compiling db_tool_monitor (unified GUI from db_tool_manager.c)..."
        gcc -Wall -Wextra -O2 -pthread \
            $(pkg-config --cflags gtk+-3.0) \
            -o db_tool_monitor db_tool_manager.c \
            $(pkg-config --libs gtk+-3.0) -pthread -lm
        print_info "    → Built with viewing AND editing capabilities"

        print_info "  Compiling db_tool_manager (same as monitor)..."
        gcc -Wall -Wextra -O2 -pthread \
            $(pkg-config --cflags gtk+-3.0) \
            -o db_tool_manager db_tool_manager.c \
            $(pkg-config --libs gtk+-3.0) -pthread -lm
        print_info "    → Required for auto-launch from db_tool"
    else
        print_warning "  Skipping GUIs (GTK3 not available)"
    fi

    # Strip binaries to reduce size
    print_info "  Stripping binaries..."
    strip db_tool 2>/dev/null || true
    strip db_tool_monitor 2>/dev/null || true
    strip db_tool_manager 2>/dev/null || true

    print_status "Compilation complete"
    cd "${SCRIPT_DIR}"
}

# Copy binaries
install_binaries() {
    print_info "Installing binaries..."

    cp "${SRC_DIR}/db_tool" "${PKG_DIR}/usr/local/bin/"
    chmod 755 "${PKG_DIR}/usr/local/bin/db_tool"

    if [ -f "${SRC_DIR}/db_tool_monitor" ]; then
        cp "${SRC_DIR}/db_tool_monitor" "${PKG_DIR}/usr/local/bin/"
        chmod 755 "${PKG_DIR}/usr/local/bin/db_tool_monitor"

        if [ -f "${SRC_DIR}/db_tool_manager" ]; then
            cp "${SRC_DIR}/db_tool_manager" "${PKG_DIR}/usr/local/bin/"
            chmod 755 "${PKG_DIR}/usr/local/bin/db_tool_manager"
            print_status "Installed db_tool, db_tool_monitor, and db_tool_manager"
        else
            print_status "Installed db_tool and db_tool_monitor"
        fi
    else
        print_warning "GUIs not built (GTK3 not available)"
        print_status "Installed db_tool only"
    fi
}

# Copy source files
install_source() {
    print_info "Installing source code..."

    cp "${SRC_DIR}/db_tool.c" "${PKG_DIR}/usr/local/share/${PKG_NAME}/src/"
    cp "${SRC_DIR}/db_tool_manager.c" "${PKG_DIR}/usr/local/share/${PKG_NAME}/src/"
    cp "${SRC_DIR}/Makefile" "${PKG_DIR}/usr/local/share/${PKG_NAME}/src/"

    print_info "  → Included unified GUI source (db_tool_manager.c)"

    # Create REBUILD.txt
    cat > "${PKG_DIR}/usr/local/share/${PKG_NAME}/src/REBUILD.txt" << 'EOF'
# Rebuilding LinuxCNC DB Tool from Source

## Requirements

sudo apt-get install build-essential libgtk-3-dev pkg-config

## Build

cd /usr/local/share/linuxcnc-db-tool/src/
make clean
make all

## Install

sudo cp db_tool db_tool_monitor /usr/local/bin/
sudo chmod +x /usr/local/bin/db_tool*

## Test

db_tool --help
echo "g" | db_tool /tmp/test.dat

## Done!

Your rebuilt binaries are now installed.
EOF

    chmod 644 "${PKG_DIR}/usr/local/share/${PKG_NAME}/src/"*

    print_status "Source code installed"
}

# Copy documentation
install_docs() {
    print_info "Installing documentation..."

    local doc_dir="${PKG_DIR}/usr/local/share/doc/${PKG_NAME}"

    # Create comprehensive documentation
    cat > "${doc_dir}/README.md" << 'EOF'
# LinuxCNC Tool Usage Tracking System

A lightweight C program that tracks tool usage time for LinuxCNC machines with tool changers.

## Features

- Real-time usage tracking - Automatically tracks how long each tool is in the spindle
- Persistent storage - Usage times saved to database file
- Low overhead - Compiled C code, no Python interpreter needed
- Thread-safe - Concurrent access protection with mutexes
- Configurable updates - Set update interval from seconds to minutes
- Random/Non-random toolchanger support
- Compatible - Works with existing LinuxCNC installations

## Quick Start

### Install System-Wide

Already installed if you're reading this!

### Configure Your Machine

Edit your LinuxCNC INI file (`~/linuxcnc/configs/your_machine/your_machine.ini`):

```ini
[EMCIO]
RANDOM_TOOLCHANGER = 0
DB_PROGRAM = /usr/local/bin/db_tool --period 1.0 --nonrandom /tmp/my_machine_tools.dat
```

### Start LinuxCNC

```bash
linuxcnc your_machine.ini
```

### Monitor Usage

```bash
watch -n 0.5 cat /tmp/my_machine_tools.dat
```

Or use the GUI:

```bash
db_tool_monitor /tmp/my_machine_tools.dat
```

## Documentation

- QUICK_INSTALL.txt - Quick installation guide
- HOW_IT_WORKS.md - Detailed explanation of the protocol
- GUI_QUICK_START.txt - GUI monitor usage
- TARGET_MACHINE_SETUP.md - Setting up on your CNC machine

## Source Code

Source code is included in `/usr/local/share/linuxcnc-db-tool/src/`

To rebuild: See src/REBUILD.txt

## License

GPL v2 - Same as LinuxCNC
EOF

    cat > "${doc_dir}/QUICK_INSTALL.txt" << 'EOF'
LinuxCNC DB Tool - Quick Installation Guide
============================================

1. CONFIGURE INI FILE

   Edit your LinuxCNC machine INI file:

   $ nano ~/linuxcnc/configs/my_machine/my_machine.ini

   Add to [EMCIO] section:

   RANDOM_TOOLCHANGER = 0
   DB_PROGRAM = db_tool --period 1.0 --nonrandom /path/to/tools.dat

   Comment out old TOOL_TABLE line:

   #TOOL_TABLE = tool.tbl

2. IMPORT EXISTING TOOLS (Optional)

   If you have an existing tool table:

   $ db_tool --import-tooltable \
       ~/linuxcnc/configs/my_machine/tool.tbl \
       ~/linuxcnc_tool_data/my_tools.dat

3. START LINUXCNC

   $ linuxcnc ~/linuxcnc/configs/my_machine/my_machine.ini

4. MONITOR USAGE

   GUI:      $ db_tool_monitor ~/linuxcnc_tool_data/my_tools.dat
   Terminal: $ watch -n 0.5 cat ~/linuxcnc_tool_data/my_tools.dat

Done!
EOF

    cat > "${doc_dir}/HOW_IT_WORKS.md" << 'EOF'
# How DB_PROGRAM and db_tool Work Together

## You Don't Run db_tool Manually!

**IMPORTANT:** LinuxCNC automatically starts and stops db_tool. You only interact with LinuxCNC.

## The Workflow

1. You start LinuxCNC
2. LinuxCNC reads INI file and sees DB_PROGRAM
3. LinuxCNC automatically launches db_tool in background
4. db_tool loads tools from database file
5. LinuxCNC requests tools from db_tool via pipes
6. db_tool sends tool data back to LinuxCNC
7. LinuxCNC displays tools in AXIS
8. You use tools normally (M6 T1, etc.)
9. db_tool tracks usage time automatically
10. You exit LinuxCNC - it stops db_tool automatically

## Communication Protocol

LinuxCNC communicates with db_tool via stdin/stdout pipes:

- `g` - Get all tools
- `l T# P#` - Load tool into spindle
- `u T# P#` - Unload tool from spindle
- `p T# P# Z# D#` - Update tool parameters

## Database File Format

```
; LinuxCNC Tool Database
; Generated: 1761698732
; CURRENT_TOOL=1
; LOAD_TIME=1761698800
;
T1  P1  Z0.5000 D3.0000 M56.817 ; 3mm end mill
T2  P2  Z0.1000 D1.5000 M12.345 ; 1.5mm end mill
```

Fields:
- T - Tool number
- P - Pocket number
- Z - Z offset (tool length)
- D - Diameter
- M - Usage time in minutes (tracked automatically)

## Spindle-Aware Timing

By default, db_tool only counts time when the spindle is running. This ensures only actual cutting time is tracked, not rapid moves or tool changes.

Disable with: `--no-spindle-aware`
EOF

    cat > "${doc_dir}/GUI_QUICK_START.txt" << 'EOF'
DB Tool Monitor - GUI Quick Start
==================================

LAUNCH GUI
----------
$ db_tool_monitor ~/linuxcnc_tool_data/my_tools.dat

Or from applications menu: "Tool Usage Monitor"

FEATURES
--------
- Real-time table display of all tools
- Green highlighting of currently loaded tool
- Usage time tracking (updates every second)
- Sortable columns (click headers)
- Auto-refresh

USAGE
-----
1. Start LinuxCNC with db_tool configured
2. Launch db_tool_monitor
3. Select your database file
4. Watch real-time updates as you use tools

The currently loaded tool will be highlighted in green.
Usage times update automatically.

TROUBLESHOOTING
---------------
Database file not found:
  - Make sure LinuxCNC is running
  - Check database path matches your INI file
  - Try: watch -n 0.5 cat /path/to/database.dat

No tool highlighted:
  - Load a tool in LinuxCNC (M6 T1)
  - Check LinuxCNC logs: ~/.linuxcnc_print.txt
EOF

    cat > "${doc_dir}/copyright" << 'EOF'
Format: https://www.debian.org/doc/packaging-manuals/copyright-format/1.0/
Upstream-Name: linuxcnc-db-tool
Source: https://linuxcnc.org

Files: *
Copyright: 2025 LinuxCNC Community
License: GPL-2+
 This package is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.
 .
 This package is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 .
 You should have received a copy of the GNU General Public License
 along with this program. If not, see <https://www.gnu.org/licenses/>
 .
 On Debian systems, the complete text of the GNU General
 Public License version 2 can be found in "/usr/share/common-licenses/GPL-2".
EOF

    echo "$(date -R)" | gzip -9 > "${doc_dir}/changelog.gz"

    chmod 644 "${doc_dir}"/*

    print_status "Documentation installed"
}

# Create helper scripts
install_scripts() {
    print_info "Installing helper scripts..."

    local script_dir="${PKG_DIR}/usr/local/share/${PKG_NAME}"

    # Create quick_setup.sh
    cat > "${script_dir}/quick_setup.sh" << 'EOF'
#!/bin/bash
# Quick setup wizard for LinuxCNC DB Tool

echo "╔══════════════════════════════════════════════════════════════════════════╗"
echo "║          LinuxCNC DB Tool - Quick Setup Wizard                           ║"
echo "╚══════════════════════════════════════════════════════════════════════════╝"
echo

# Search for LinuxCNC configs in multiple locations
echo "Searching for LinuxCNC configurations..."
echo

# Possible config locations
SEARCH_PATHS=(
    "$HOME/linuxcnc/configs"
    "/usr/share/doc/linuxcnc/examples/sample-configs"
    "/usr/local/share/doc/linuxcnc/examples/sample-configs"
)

echo "Search locations:"
for path in "${SEARCH_PATHS[@]}"; do
    if [ -d "$path" ]; then
        echo "  ✓ $path"
    else
        echo "  ✗ $path (not found)"
    fi
done
echo

# Find all configs
declare -a configs
declare -a config_locations

for search_path in "${SEARCH_PATHS[@]}"; do
    if [ -d "$search_path" ]; then
        while IFS= read -r -d '' config_dir; do
            # Skip the parent "configs" or "sample-configs" directory itself
            basename="${config_dir##*/}"
            if [ "$basename" != "configs" ] && [ "$basename" != "sample-configs" ]; then
                configs+=("$config_dir")
                # Determine location label
                if [[ "$config_dir" == "$HOME"* ]]; then
                    config_locations+=("User")
                else
                    config_locations+=("System")
                fi
            fi
        done < <(find "$search_path" -maxdepth 1 -type d -print0 2>/dev/null | sort -z)
    fi
done

# Check if any configs found
if [ ${#configs[@]} -eq 0 ]; then
    echo "Error: No LinuxCNC configurations found!"
    echo
    echo "Searched in:"
    for path in "${SEARCH_PATHS[@]}"; do
        echo "  - $path"
    done
    echo
    echo "Please ensure LinuxCNC is installed and configurations exist."
    exit 1
fi

# Display found configs
echo "Found ${#configs[@]} configuration(s):"
echo

for i in "${!configs[@]}"; do
    basename="${configs[$i]##*/}"
    location="${config_locations[$i]}"
    printf "  %2d. %-40s [%s]\n" "$((i+1))" "$basename" "$location"
done
echo

# Get user selection
read -p "Select configuration number (1-${#configs[@]}): " config_num

if [ -z "$config_num" ] || [ "$config_num" -lt 1 ] || [ "$config_num" -gt ${#configs[@]} ]; then
    echo "Invalid selection"
    exit 1
fi

selected_config="${configs[$((config_num-1))]}"
selected_location="${config_locations[$((config_num-1))]}"
config_name=$(basename "$selected_config")

echo
echo "Selected: $config_name [$selected_location]"
echo "Path: $selected_config"
echo

# Database location
# Sanitize config name: replace periods, spaces, and special chars with underscores
DB_DIR="$HOME/linuxcnc_tool_data"
safe_config_name=$(echo "$config_name" | tr '. ' '_')
DB_FILE="$DB_DIR/${safe_config_name}_tools.dat"

echo "Database will be saved to:"
echo "  $DB_FILE"
echo

read -p "Press Enter to continue or Ctrl-C to cancel..."

# Create database directory
mkdir -p "$DB_DIR"

# Look for existing tool table
echo
echo "Searching for tool table files..."
TOOL_TABLES=($(find "$selected_config" -name "*.tbl" -type f 2>/dev/null))

if [ ${#TOOL_TABLES[@]} -gt 0 ]; then
    echo "Found ${#TOOL_TABLES[@]} tool table(s):"
    for i in "${!TOOL_TABLES[@]}"; do
        echo "  $((i+1)). $(basename ${TOOL_TABLES[$i]})"
    done
    echo

    if [ ${#TOOL_TABLES[@]} -eq 1 ]; then
        # Only one tool table found
        TOOL_TABLE="${TOOL_TABLES[0]}"
        echo "Using: $(basename $TOOL_TABLE)"
        read -p "Import tools from this file? (Y/n): " import_tools
    else
        # Multiple tool tables found
        read -p "Select tool table number to import (1-${#TOOL_TABLES[@]}) or 0 to skip: " tbl_num

        if [ -n "$tbl_num" ] && [ "$tbl_num" -ge 1 ] && [ "$tbl_num" -le ${#TOOL_TABLES[@]} ]; then
            TOOL_TABLE="${TOOL_TABLES[$((tbl_num-1))]}"
            import_tools="y"
        else
            import_tools="n"
        fi
    fi

    if [ "$import_tools" != "n" ] && [ "$import_tools" != "N" ] && [ -n "$TOOL_TABLE" ]; then
        echo
        echo "Importing tools from: $(basename $TOOL_TABLE)"
        if db_tool --import-tooltable "$TOOL_TABLE" "$DB_FILE" 2>&1; then
            echo "✓ Tools imported successfully!"
        else
            echo "✗ Error importing tools. You can import manually later."
        fi
    fi
else
    echo "No existing tool tables found."
    echo "A default database will be created when db_tool starts."
fi

echo
echo "╔══════════════════════════════════════════════════════════════════════════╗"
echo "║                          Setup Complete!                                 ║"
echo "╚══════════════════════════════════════════════════════════════════════════╝"
echo
echo "Next steps:"
echo
echo "1. Find your INI file in:"
echo "   $selected_config/"
echo
echo "2. Edit the INI file (example: axis_mm.ini):"
echo "   nano $selected_config/*.ini"
echo
echo "3. Add to [EMCIO] section:"
echo "   RANDOM_TOOLCHANGER = 0"
echo "   DB_PROGRAM = db_tool --period 1.0 --nonrandom $DB_FILE"
echo
echo "4. Comment out old TOOL_TABLE line:"
echo "   #TOOL_TABLE = tool.tbl"
echo
echo "5. Start LinuxCNC and test:"
echo "   linuxcnc $selected_config/<your-config>.ini"
echo
echo "6. Monitor tool usage:"
echo "   db_tool_monitor $DB_FILE"
echo
echo "Database file: $DB_FILE"
echo
EOF

    chmod 755 "${script_dir}/quick_setup.sh"

    # Create discover_linuxcnc.sh
    cat > "${script_dir}/discover_linuxcnc.sh" << 'EOF'
#!/bin/bash
# Discover LinuxCNC installation paths

echo "╔══════════════════════════════════════════════════════════════════════════╗"
echo "║              LinuxCNC Installation Discovery                             ║"
echo "╚══════════════════════════════════════════════════════════════════════════╝"
echo

# Check for LinuxCNC binary
echo "LinuxCNC Installation"
echo "────────────────────────────────────────────────────────────────────────────"
LINUXCNC_BIN=$(which linuxcnc 2>/dev/null)
if [ -n "$LINUXCNC_BIN" ]; then
    echo "Binary:  $LINUXCNC_BIN"
    LINUXCNC_VERSION=$(linuxcnc -v 2>&1 | head -1)
    echo "Version: $LINUXCNC_VERSION"
else
    echo "Binary:  NOT FOUND"
fi
echo

# Check for configs in multiple locations
echo "Configuration Directories"
echo "────────────────────────────────────────────────────────────────────────────"

SEARCH_PATHS=(
    "$HOME/linuxcnc/configs"
    "/usr/share/doc/linuxcnc/examples/sample-configs"
    "/usr/local/share/doc/linuxcnc/examples/sample-configs"
)

total_configs=0

for search_path in "${SEARCH_PATHS[@]}"; do
    if [ -d "$search_path" ]; then
        configs_found=$(find "$search_path" -maxdepth 1 -type d -not -name "configs" -not -name "sample-configs" 2>/dev/null | wc -l)

        if [ "$configs_found" -gt 0 ]; then
            echo "Location: $search_path"
            echo "Found $configs_found configuration(s):"

            find "$search_path" -maxdepth 1 -type d -not -name "configs" -not -name "sample-configs" 2>/dev/null | sort | while read dir; do
                config_name=$(basename "$dir")
                # Count INI files
                ini_count=$(find "$dir" -maxdepth 1 -name "*.ini" -type f 2>/dev/null | wc -l)
                echo "  ✓ $config_name ($ini_count INI file(s))"
            done
            echo
            total_configs=$((total_configs + configs_found))
        fi
    fi
done

if [ "$total_configs" -eq 0 ]; then
    echo "No configurations found in standard locations."
    echo
fi

# Check for tool tables
echo "Tool Tables"
echo "────────────────────────────────────────────────────────────────────────────"

tool_tables_found=0

for search_path in "${SEARCH_PATHS[@]}"; do
    if [ -d "$search_path" ]; then
        tables=$(find "$search_path" -name "*.tbl" -type f 2>/dev/null)
        if [ -n "$tables" ]; then
            echo "$tables" | while read tbl; do
                config_dir=$(dirname "$tbl")
                config_name=$(basename "$config_dir")
                tbl_name=$(basename "$tbl")
                echo "  ✓ $config_name/$tbl_name"
                tool_tables_found=$((tool_tables_found + 1))
            done
        fi
    fi
done

if [ "$tool_tables_found" -eq 0 ]; then
    echo "No tool tables found."
fi
echo

# Check for db_tool installation
echo "DB Tool Installation"
echo "────────────────────────────────────────────────────────────────────────────"
DB_TOOL_BIN=$(which db_tool 2>/dev/null)
if [ -n "$DB_TOOL_BIN" ]; then
    echo "Binary:  $DB_TOOL_BIN"

    # Check if package is installed
    if dpkg -l linuxcnc-db-tool >/dev/null 2>&1; then
        PKG_VERSION=$(dpkg -l linuxcnc-db-tool | grep '^ii' | awk '{print $3}')
        echo "Package: linuxcnc-db-tool $PKG_VERSION (installed)"
    fi

    # Check for source
    if [ -d "/usr/local/share/linuxcnc-db-tool/src" ]; then
        echo "Source:  /usr/local/share/linuxcnc-db-tool/src/"
    fi
else
    echo "Binary:  NOT FOUND"
    echo "To install: sudo dpkg -i linuxcnc-db-tool_*.deb"
fi
echo

# Check for existing databases
echo "Tool Usage Databases"
echo "────────────────────────────────────────────────────────────────────────────"

DB_DIR="$HOME/linuxcnc_tool_data"
if [ -d "$DB_DIR" ]; then
    db_count=$(find "$DB_DIR" -name "*.dat" -type f 2>/dev/null | wc -l)
    if [ "$db_count" -gt 0 ]; then
        echo "Location: $DB_DIR"
        echo "Found $db_count database(s):"
        find "$DB_DIR" -name "*.dat" -type f 2>/dev/null | while read db; do
            db_name=$(basename "$db")
            db_size=$(du -h "$db" | cut -f1)
            tool_count=$(grep -c "^T" "$db" 2>/dev/null || echo "0")
            echo "  ✓ $db_name ($db_size, $tool_count tools)"
        done
    else
        echo "Location: $DB_DIR (empty)"
    fi
else
    echo "No databases found."
    echo "Databases will be created at: $DB_DIR"
fi
echo

echo "╔══════════════════════════════════════════════════════════════════════════╗"
echo "║                          Discovery Complete                              ║"
echo "╚══════════════════════════════════════════════════════════════════════════╝"
echo
echo "Total configurations found: $total_configs"
echo
echo "To set up db_tool for a configuration, run:"
echo "  /usr/local/share/linuxcnc-db-tool/quick_setup.sh"
echo
EOF

    chmod 755 "${script_dir}/discover_linuxcnc.sh"

    print_status "Helper scripts installed"
}

# Create example configuration
install_examples() {
    print_info "Installing example configuration..."

    local example_dir="${PKG_DIR}/usr/local/share/${PKG_NAME}/examples"

    cat > "${example_dir}/db_c_test.ini" << 'EOF'
# Example LinuxCNC INI configuration for db_tool
# This demonstrates how to configure db_tool in your machine INI file

[EMC]
VERSION = 1.1
MACHINE = LinuxCNC-DB-Tool-Example

[EMCIO]
EMCIO = io
CYCLE_TIME = 0.100

# Toolchanger type: 0 = non-random (sequential), 1 = random (ATC)
RANDOM_TOOLCHANGER = 0

# Tool database program with usage tracking
# Update period: 1.0 minutes (change as needed)
# --nonrandom: non-random toolchanger (change to --random for ATC)
# Database path: change to your desired location
DB_PROGRAM = db_tool --period 1.0 --nonrandom /tmp/example_tools.dat

# Old static tool table (comment out when using DB_PROGRAM)
#TOOL_TABLE = tool.tbl

# Tool change position (optional)
TOOL_CHANGE_POSITION = 0 0 50

# Alternative configurations:
# Fast updates (30 seconds):
#   DB_PROGRAM = db_tool --period 0.5 --nonrandom /tmp/tools.dat
#
# Random toolchanger (ATC):
#   RANDOM_TOOLCHANGER = 1
#   DB_PROGRAM = db_tool --period 1.0 --random /tmp/tools.dat
#
# Legacy mode (count all time, not just spindle-on):
#   DB_PROGRAM = db_tool --no-spindle-aware --period 1.0 /tmp/tools.dat
EOF

    chmod 644 "${example_dir}/db_c_test.ini"

    print_status "Example configuration installed"
}

# Create desktop file for GUI
install_desktop_file() {
    print_info "Installing desktop application entry..."

    cat > "${PKG_DIR}/usr/local/share/applications/db_tool_monitor.desktop" << 'EOF'
[Desktop Entry]
Version=1.0
Type=Application
Name=LinuxCNC Tool Manager
Comment=View and edit LinuxCNC tool database with usage tracking
Exec=db_tool_monitor
Icon=applications-engineering
Terminal=false
Categories=Science;Engineering;
Keywords=linuxcnc;cnc;tools;monitor;manager;editor;
EOF

    chmod 644 "${PKG_DIR}/usr/local/share/applications/db_tool_monitor.desktop"

    print_status "Desktop application entry installed"
}

# Create DEBIAN control file
create_control() {
    print_info "Creating DEBIAN control file..."

    cat > "${PKG_DIR}/DEBIAN/control" << EOF
Package: ${PKG_NAME}
Version: ${PKG_VERSION}
Section: science
Priority: optional
Architecture: ${PKG_ARCH}
Depends: libc6 (>= 2.17), libpthread-stubs0-dev, libgtk-3-0
Suggests: linuxcnc (>= 2.8), libgtk-3-dev
Installed-Size: 336
Maintainer: ${PKG_MAINTAINER}
Description: Tool usage time tracking and management for LinuxCNC
 A lightweight C program that tracks tool usage time for LinuxCNC machines
 with a unified GTK3 GUI for viewing and editing (no Python dependencies).
 .
 Features:
  - Real-time usage tracking
  - Automatic GUI launch when LinuxCNC starts
  - Native C GTK3 GUI with viewing AND editing capabilities
  - Add, delete, and modify tools directly in the GUI
  - Edit all LinuxCNC tool parameters (offsets, diameter, etc.)
  - Current tool highlighting
  - Persistent storage of usage times
  - GUI reconnection to running LinuxCNC instance
  - Automatic LinuxCNC path discovery
  - Interactive setup wizard
  - Low memory footprint (~140KB total)
  - Thread-safe operation
  - Random and non-random toolchanger support
  - Source code included for local compilation
  - Configurable update intervals
 .
 This package provides the db_tool program for LinuxCNC's DB_PROGRAM
 interface, enabling automatic tracking of tool cutting time.
 .
 To rebuild from source, install libgtk-3-dev and run make in
 /usr/local/share/linuxcnc-db-tool/src/
 .
 Source code and Makefile are included for rebuilding on systems where
 the precompiled binary doesn't work (architecture mismatch, etc).
Homepage: https://linuxcnc.org
EOF

    chmod 644 "${PKG_DIR}/DEBIAN/control"

    print_status "Control file created"
}

# Create postinst script
create_postinst() {
    print_info "Creating post-installation script..."

    cat > "${PKG_DIR}/DEBIAN/postinst" << 'EOF'
#!/bin/bash
# Post-installation script for linuxcnc-db-tool

set -e

echo "═══════════════════════════════════════════════════════════"
echo "  LinuxCNC Tool Usage Tracker - Installation"
echo "═══════════════════════════════════════════════════════════"
echo

# Cleanup old instances
echo "Cleaning up old instances..."
pkill -TERM -x db_tool 2>/dev/null || true
pkill -TERM -x db_tool_monitor 2>/dev/null || true
sleep 1

# Detect actual user
ACTUAL_USER="${SUDO_USER:-${USER}}"
if [ "$ACTUAL_USER" = "root" ]; then
    ACTUAL_USER=$(who | awk '{print $1}' | sort -u | grep -v root | head -n1)
fi
if [ -z "$ACTUAL_USER" ]; then
    ACTUAL_USER="${USER}"
fi

USER_HOME=$(eval echo ~${ACTUAL_USER})
DEFAULT_DB_DIR="${USER_HOME}/linuxcnc_tool_data"
DEFAULT_DB_FILE="${DEFAULT_DB_DIR}/tool_usage.dat"

echo "Installed files:"
echo "  Binary:        /usr/local/bin/db_tool"
echo "  GUI Monitor:   /usr/local/bin/db_tool_monitor"
echo "  GUI Manager:   /usr/local/bin/db_tool_manager"
echo "  Documentation: /usr/local/share/doc/linuxcnc-db-tool/"
echo "  Source code:   /usr/local/share/linuxcnc-db-tool/src/"
echo

# Interactive setup
if [ -t 0 ]; then
    echo "─────────────────────────────────────────────────────────"
    echo "Quick Setup"
    echo "─────────────────────────────────────────────────────────"
    echo
    read -p "Run automatic setup wizard now? (Y/n): " RUN_SETUP
    echo

    if [ -z "$RUN_SETUP" ] || [ "$RUN_SETUP" = "Y" ] || [ "$RUN_SETUP" = "y" ]; then
        cd /usr/local/share/linuxcnc-db-tool
        if [ "$ACTUAL_USER" != "root" ] && [ "$ACTUAL_USER" != "$USER" ]; then
            sudo -u "$ACTUAL_USER" bash ./quick_setup.sh
        else
            bash ./quick_setup.sh
        fi
    else
        echo "To run setup later:"
        echo "  cd /usr/local/share/linuxcnc-db-tool"
        echo "  ./quick_setup.sh"
        echo
    fi
fi

echo "─────────────────────────────────────────────────────────"
echo "Documentation:"
echo "─────────────────────────────────────────────────────────"
echo "  Quick start: /usr/local/share/doc/linuxcnc-db-tool/QUICK_INSTALL.txt"
echo "  Rebuild:     /usr/local/share/linuxcnc-db-tool/src/REBUILD.txt"
echo "  Help:        db_tool --help"
echo
echo "═══════════════════════════════════════════════════════════"

exit 0
EOF

    chmod 755 "${PKG_DIR}/DEBIAN/postinst"

    print_status "Post-installation script created"
}

# Create prerm script
create_prerm() {
    print_info "Creating pre-removal script..."

    cat > "${PKG_DIR}/DEBIAN/prerm" << 'EOF'
#!/bin/bash
# Pre-removal script for linuxcnc-db-tool

set -e

echo "Stopping db_tool processes..."
pkill -TERM -x db_tool 2>/dev/null || true
pkill -TERM -x db_tool_monitor 2>/dev/null || true
pkill -TERM -x db_tool_manager 2>/dev/null || true
sleep 2
pkill -KILL -x db_tool 2>/dev/null || true

echo "Note: Database files are NOT removed."
echo "Your tool usage data is preserved at:"
echo "  ~/linuxcnc_tool_data/"
echo

exit 0
EOF

    chmod 755 "${PKG_DIR}/DEBIAN/prerm"

    print_status "Pre-removal script created"
}

# Set proper permissions
set_permissions() {
    print_info "Setting file permissions..."

    # Files: 644
    find "${PKG_DIR}" -type f -exec chmod 644 {} \;

    # Directories: 755
    find "${PKG_DIR}" -type d -exec chmod 755 {} \;

    # Binaries: 755
    chmod 755 "${PKG_DIR}/usr/local/bin/"* 2>/dev/null || true

    # Scripts: 755
    chmod 755 "${PKG_DIR}/DEBIAN/postinst" "${PKG_DIR}/DEBIAN/prerm" 2>/dev/null || true
    chmod 755 "${PKG_DIR}/usr/local/share/${PKG_NAME}/"*.sh 2>/dev/null || true

    print_status "Permissions set"
}

# Build the package
build_package() {
    print_info "Building Debian package..."

    cd "${BUILD_DIR}"
    dpkg-deb --build "${PKG_NAME}_${PKG_VERSION}_${PKG_ARCH}"

    if [ $? -eq 0 ]; then
        print_status "Package built successfully"
    else
        print_error "Package build failed"
        exit 1
    fi

    cd "${SCRIPT_DIR}"
}

# Move package to output
finalize() {
    print_info "Finalizing package..."

    mv "${BUILD_DIR}/${PKG_NAME}_${PKG_VERSION}_${PKG_ARCH}.deb" "${OUTPUT_DIR}/"

    local pkg_file="${OUTPUT_DIR}/${PKG_NAME}_${PKG_VERSION}_${PKG_ARCH}.deb"
    local pkg_size=$(du -h "$pkg_file" | cut -f1)

    print_status "Package created: ${pkg_file}"
    print_info "Package size: ${pkg_size}"

    # Generate checksums
    print_info "Generating checksums..."
    cd "${OUTPUT_DIR}"
    md5sum "${PKG_NAME}_${PKG_VERSION}_${PKG_ARCH}.deb" > "${PKG_NAME}_${PKG_VERSION}_${PKG_ARCH}.deb.md5"
    sha256sum "${PKG_NAME}_${PKG_VERSION}_${PKG_ARCH}.deb" > "${PKG_NAME}_${PKG_VERSION}_${PKG_ARCH}.deb.sha256"

    print_status "Checksums generated"

    # Show checksums
    echo
    echo -e "${BLUE}MD5:${NC}"
    cat "${PKG_NAME}_${PKG_VERSION}_${PKG_ARCH}.deb.md5"
    echo
    echo -e "${BLUE}SHA256:${NC}"
    cat "${PKG_NAME}_${PKG_VERSION}_${PKG_ARCH}.deb.sha256"
    echo
}

# Cleanup build directory
cleanup() {
    print_info "Cleaning up build directory..."
    rm -rf "${BUILD_DIR}"
    print_status "Cleanup complete"
}

# Display final instructions
show_instructions() {
    echo
    echo -e "${GREEN}╔══════════════════════════════════════════════════════════════════════════╗${NC}"
    echo -e "${GREEN}║                    BUILD COMPLETE!                                       ║${NC}"
    echo -e "${GREEN}╚══════════════════════════════════════════════════════════════════════════╝${NC}"
    echo
    echo -e "${BLUE}Package:${NC} ${PKG_NAME}_${PKG_VERSION}_${PKG_ARCH}.deb"
    echo -e "${BLUE}Location:${NC} ${OUTPUT_DIR}"
    echo
    echo -e "${YELLOW}To install:${NC}"
    echo "  sudo dpkg -i ${PKG_NAME}_${PKG_VERSION}_${PKG_ARCH}.deb"
    echo
    echo -e "${YELLOW}To test:${NC}"
    echo "  dpkg -I ${PKG_NAME}_${PKG_VERSION}_${PKG_ARCH}.deb"
    echo "  dpkg -c ${PKG_NAME}_${PKG_VERSION}_${PKG_ARCH}.deb"
    echo
    echo -e "${YELLOW}To verify:${NC}"
    echo "  md5sum -c ${PKG_NAME}_${PKG_VERSION}_${PKG_ARCH}.deb.md5"
    echo "  sha256sum -c ${PKG_NAME}_${PKG_VERSION}_${PKG_ARCH}.deb.sha256"
    echo
}

# Main build process
main() {
    check_dependencies
    clean_build
    create_structure
    compile_binaries
    install_binaries
    install_source
    install_docs
    install_scripts
    install_examples
    install_desktop_file
    create_control
    create_postinst
    create_prerm
    set_permissions
    build_package
    finalize
    cleanup
    show_instructions
}

# Run main
main

exit 0
EOF
