/*
 * LinuxCNC Tool Database Program in C
 * Implements tool usage time tracking and database persistence
 *
 * Communication Protocol:
 *   - Startup: Send "v2.1\n" to stdout
 *   - Commands from LinuxCNC on stdin:
 *     g           - Get all tools
 *     p <toolline> - Put/update tool parameters
 *     l <toolline> - Load tool into spindle
 *     u <toolline> - Unload tool from spindle
 *   - Responses to stdout (always end with \n and fflush)
 *
 * Usage: db_tool [options] <database_file>
 *   Options:
 *     --period <minutes>       : Update period (default: 1.0)
 *     --random                 : Random toolchanger mode
 *     --nonrandom              : Non-random toolchanger mode (default)
 *     --import-tooltable <file>: Import existing tool.tbl file
 *     --help                   : Show help
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <pthread.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>
#include <math.h>
#include <stdarg.h>
#include <stdatomic.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <poll.h>

#define DB_VERSION "v2.1"
#define MAX_TOOLS 200
#define LINE_LEN 512
#define TOOLNO_MIN 1
#define TOOLNO_MAX 99

typedef struct {
    int toolno;
    int pocketno;
    double x, y, z;           // XYZ offsets
    double a, b, c;           // ABC rotary offsets
    double u, v, w;           // UVW offsets
    double i, j, q;           // IJQ offsets
    double diameter;          // Tool diameter
    double frontangle;        // Front angle
    double backangle;         // Back angle
    int orientation;          // Tool orientation
    double usage_seconds;     // Accumulated cutting time in seconds
    time_t load_time;         // When tool was loaded (0 = not loaded)
    int in_use;              // 1 if slot is used, 0 if empty
    // New tracking fields (for compatibility with db_tool_manager)
    char uid[64];            // Unique instance ID
    char type[64];           // Tool type (ENDMILL, DRILL, etc.)
    char status[16];         // ACTIVE or RETIRED
    time_t created;          // When tool was created
} Tool;

// Global state
static Tool tools[MAX_TOOLS];
static int tool_count = 0;
static int spindle_tool_idx = -1;  // Index into tools array, -1 = no tool
static int random_toolchanger = 0;
static char db_file[512] = "/tmp/db_tool_file";
static double period_minutes = 1.0;
static pthread_mutex_t tools_mutex = PTHREAD_MUTEX_INITIALIZER;
static _Atomic int running = 1;  // Atomic for safe multi-thread access
static volatile sig_atomic_t got_sigterm = 0;  // Async-signal-safe shutdown flag
static FILE *logfile = NULL;  // stderr for logging
static int spindle_aware = 1;  // Count time only when spindle is running (default: enabled)
static time_t last_spindle_check = 0;  // Last time we checked spindle state
static int last_spindle_state = 0;  // Last known spindle state (0=off, 1=on)
static time_t last_file_mtime = 0;  // Last known modification time of database file

// File version tracking (protected by tools_mutex) for detecting concurrent modifications
static struct {
    time_t mtime;
    off_t size;
} db_version = {0, 0};

static int launch_gui = 1;  // Launch GUI manager on startup (default: enabled)
static pid_t gui_pid = 0;  // PID of launched GUI process
static pthread_mutex_t log_mutex = PTHREAD_MUTEX_INITIALIZER;  // Protect log output

// Function prototypes
void init_tools(void);
void load_tools_from_file(const char *filename);
void save_tools_to_file(const char *filename, const char *comment);
void send_version(void);
void send_tool(Tool *t);
void handle_get(void);
void handle_put(char *params);
void handle_load(char *params);
void handle_unload(char *params);
void parse_command(char *line);
void start_tool_timer(int idx);
void stop_tool_timer(int idx);
void update_tool_time(int idx);
void pause_tool_timer(int idx);
void resume_tool_timer(int idx);
int check_spindle_running(void);
int check_cutting_active(void);
void *periodic_task(void *arg);
void *command_loop(void *arg);
void signal_handler(int signo);
int find_tool_by_number(int toolno);
void parse_tool_line(const char *line, Tool *t);
void parse_tool_line_merge(const char *line, Tool *t);
void generate_uid(char *uid, size_t len);
void log_msg(const char *fmt, ...);
void launch_gui_manager(void);

// Logging function with timestamp (thread-safe with mutex)
void log_msg(const char *fmt, ...) {
    va_list args;
    va_start(args, fmt);
    pthread_mutex_lock(&log_mutex);
    if (logfile) {
        // Add timestamp to all log entries (thread-safe localtime_r)
        time_t now = time(NULL);
        struct tm tm_buf;
        struct tm *tm = localtime_r(&now, &tm_buf);
        if (tm) {
            fprintf(logfile, "[%04d-%02d-%02d %02d:%02d:%02d] db_tool: ",
                    tm->tm_year + 1900, tm->tm_mon + 1, tm->tm_mday,
                    tm->tm_hour, tm->tm_min, tm->tm_sec);
        }
        vfprintf(logfile, fmt, args);
        fprintf(logfile, "\n");
        fflush(logfile);
    }
    pthread_mutex_unlock(&log_mutex);
    va_end(args);
}

// Get file version (mtime + size) - returns 0 on success, -1 on error
static int get_file_version(const char *filename, time_t *mtime, off_t *size) {
    struct stat st;
    if (stat(filename, &st) != 0) {
        return -1;
    }
    *mtime = st.st_mtime;
    *size = st.st_size;
    return 0;
}

// Find tool index by tool number
int find_tool_by_number(int toolno) {
    for (int i = 0; i < MAX_TOOLS; i++) {
        if (tools[i].in_use && tools[i].toolno == toolno) {
            return i;
        }
    }
    return -1;
}

// Initialize tools with default values
void init_tools(void) {
    memset(tools, 0, sizeof(tools));
    tool_count = 0;
    srand(time(NULL));  // Seed random for UID generation

    // Create initial tools (T1-T20 for simulation)
    for (int tno = 1; tno <= 20; tno++) {
        int idx = tno - 1;
        tools[idx].in_use = 1;
        tools[idx].toolno = tno;
        tools[idx].pocketno = random_toolchanger ? (tno + 100) : tno;
        tools[idx].z = tno / 100.0;
        tools[idx].diameter = tno / 100.0;
        tools[idx].usage_seconds = 0.0;
        tools[idx].load_time = 0;

        // Initialize tracking fields for new tools
        generate_uid(tools[idx].uid, sizeof(tools[idx].uid));
        strcpy(tools[idx].type, "UNKNOWN");
        strcpy(tools[idx].status, "ACTIVE");
        tools[idx].created = time(NULL);

        tool_count++;
    }

    // Add T0 for random toolchanger (empty pocket)
    if (random_toolchanger) {
        tools[tool_count].in_use = 1;
        tools[tool_count].toolno = 0;
        tools[tool_count].pocketno = 0;
        generate_uid(tools[tool_count].uid, sizeof(tools[tool_count].uid));
        strcpy(tools[tool_count].type, "UNKNOWN");
        strcpy(tools[tool_count].status, "ACTIVE");
        tools[tool_count].created = time(NULL);
        tool_count++;
    }

    log_msg("Initialized %d tools (toolno %d-%d)", tool_count, TOOLNO_MIN, TOOLNO_MAX);
    save_tools_to_file(db_file, "Created database (init_tools)");

    // Print initialized tools to screen for user feedback
    log_msg("=== Tool Database Created ===");
    log_msg("%-5s %-7s %-10s %-10s %s", "Tool", "Pocket", "Z-Offset", "Diameter", "Usage (sec)");
    log_msg("%-5s %-7s %-10s %-10s %s", "----", "------", "--------", "--------", "-----------");

    for (int i = 0; i < MAX_TOOLS; i++) {
        if (!tools[i].in_use) continue;

        Tool *t = &tools[i];
        log_msg("T%-3d  P%-5d  Z%-9.4f  D%-9.4f  %.1f sec",
                t->toolno, t->pocketno, t->z, t->diameter, t->usage_seconds);
    }

    log_msg("=============================");
}

// Parse a tool line from file or LinuxCNC
void parse_tool_line(const char *line, Tool *t) {
    char temp[LINE_LEN];
    strncpy(temp, line, sizeof(temp) - 1);
    temp[sizeof(temp) - 1] = '\0';

    // Remove comments (semicolon and beyond)
    char *comment = strchr(temp, ';');
    if (comment) *comment = '\0';

    // Initialize to zero
    memset(t, 0, sizeof(Tool));
    t->in_use = 1;

    // Set defaults for new tracking fields
    strcpy(t->status, "ACTIVE");
    strcpy(t->type, "UNKNOWN");
    strcpy(t->uid, "LEGACY");

    // Parse letter-value pairs and new field formats
    char *token = strtok(temp, " \t\n");
    while (token != NULL) {
        // Check for new field formats first: UID:xxx, TYPE:xxx, STATUS:xxx, CREATED:xxx
        if (strncmp(token, "UID:", 4) == 0) {
            strncpy(t->uid, token + 4, sizeof(t->uid) - 1);
            t->uid[sizeof(t->uid) - 1] = '\0';
        } else if (strncmp(token, "TYPE:", 5) == 0) {
            strncpy(t->type, token + 5, sizeof(t->type) - 1);
            t->type[sizeof(t->type) - 1] = '\0';
        } else if (strncmp(token, "STATUS:", 7) == 0) {
            strncpy(t->status, token + 7, sizeof(t->status) - 1);
            t->status[sizeof(t->status) - 1] = '\0';
        } else if (strncmp(token, "CREATED:", 8) == 0) {
            t->created = (time_t)atol(token + 8);
        } else {
            // Traditional single-letter fields
            char letter = token[0];
            double value = 0.0;
            int int_value = 0;

            if (strlen(token) > 1) {
                if (letter == 'T' || letter == 'P') {
                    int_value = atoi(token + 1);
                } else {
                    value = atof(token + 1);
                }
            }

            switch (letter) {
                case 'T': case 't': t->toolno = int_value; break;
                case 'P': case 'p': t->pocketno = int_value; break;
                case 'X': case 'x': t->x = value; break;
            case 'Y': case 'y': t->y = value; break;
            case 'Z': case 'z': t->z = value; break;
            case 'A': case 'a': t->a = value; break;
            case 'B': case 'b': t->b = value; break;
            case 'C': case 'c': t->c = value; break;
            case 'U': case 'u': t->u = value; break;
            case 'V': case 'v': t->v = value; break;
            case 'W': case 'w': t->w = value; break;
            case 'I': case 'i': t->i = value; break;
            case 'J': case 'j': t->j = value; break;
            case 'Q': case 'q': t->q = value; break;
            case 'D': case 'd': t->diameter = value; break;
                case 'M': case 'm': t->usage_seconds = value; break;
            }
        }

        token = strtok(NULL, " \t\n");
    }
}

// Generate unique tool instance ID
void generate_uid(char *uid, size_t len) {
    time_t now = time(NULL);
    struct tm tm_buf;
    struct tm *tm_info = localtime_r(&now, &tm_buf);
    if (tm_info) {
        snprintf(uid, len, "%04d%02d%02d%02d%02d%02d-%03d",
                 tm_info->tm_year + 1900, tm_info->tm_mon + 1, tm_info->tm_mday,
                 tm_info->tm_hour, tm_info->tm_min, tm_info->tm_sec,
                 rand() % 1000);
    } else {
        snprintf(uid, len, "LEGACY");
    }
}

// Parse tool line and MERGE with existing tool (preserves unmentioned fields)
// This is critical for handle_put where LinuxCNC sends partial updates
void parse_tool_line_merge(const char *line, Tool *t) {
    char temp[LINE_LEN];
    strncpy(temp, line, sizeof(temp) - 1);
    temp[sizeof(temp) - 1] = '\0';

    // Remove comments (semicolon and beyond)
    char *comment = strchr(temp, ';');
    if (comment) *comment = '\0';

    // Parse letter-value pairs and new field formats
    // IMPORTANT: We DON'T memset here - we merge into existing tool
    char *token = strtok(temp, " \t\n");
    while (token != NULL) {
        // Check for new field formats first: UID:xxx, TYPE:xxx, STATUS:xxx, CREATED:xxx
        if (strncmp(token, "UID:", 4) == 0) {
            strncpy(t->uid, token + 4, sizeof(t->uid) - 1);
            t->uid[sizeof(t->uid) - 1] = '\0';
        } else if (strncmp(token, "TYPE:", 5) == 0) {
            strncpy(t->type, token + 5, sizeof(t->type) - 1);
            t->type[sizeof(t->type) - 1] = '\0';
        } else if (strncmp(token, "STATUS:", 7) == 0) {
            strncpy(t->status, token + 7, sizeof(t->status) - 1);
            t->status[sizeof(t->status) - 1] = '\0';
        } else if (strncmp(token, "CREATED:", 8) == 0) {
            t->created = (time_t)atol(token + 8);
        } else if (strlen(token) > 1) {
            // Traditional single-letter fields
            char letter = token[0];
            double value = 0.0;
            int int_value = 0;

            if (letter == 'T' || letter == 'P') {
                int_value = atoi(token + 1);
            } else {
                value = atof(token + 1);
            }

            switch (letter) {
                case 'T': case 't': t->toolno = int_value; break;
                case 'P': case 'p': t->pocketno = int_value; break;
                case 'X': case 'x': t->x = value; break;
                case 'Y': case 'y': t->y = value; break;
                case 'Z': case 'z': t->z = value; break;
                case 'A': case 'a': t->a = value; break;
                case 'B': case 'b': t->b = value; break;
                case 'C': case 'c': t->c = value; break;
                case 'U': case 'u': t->u = value; break;
                case 'V': case 'v': t->v = value; break;
                case 'W': case 'w': t->w = value; break;
                case 'I': case 'i': t->i = value; break;
                case 'J': case 'j': t->j = value; break;
                case 'Q': case 'q': t->q = value; break;
                case 'D': case 'd': t->diameter = value; break;
                case 'M': case 'm': t->usage_seconds = value; break;
            }
        }

        token = strtok(NULL, " \t\n");
    }
}

// Load tools from file
void load_tools_from_file(const char *filename) {
    FILE *f = fopen(filename, "r");
    if (!f) {
        log_msg("No existing database file, creating new one");
        init_tools();
        return;
    }

    char line[LINE_LEN];
    tool_count = 0;
    memset(tools, 0, sizeof(tools));

    while (fgets(line, sizeof(line), f)) {
        // Skip empty lines and comments
        if (line[0] == '\n' || line[0] == ';' || line[0] == '#') {
            continue;
        }

        Tool t;
        parse_tool_line(line, &t);

        if (t.toolno >= 0 && tool_count < MAX_TOOLS) {
            tools[tool_count] = t;
            tools[tool_count].in_use = 1;
            // IMPORTANT: Clear load_time on startup - don't continue timing from previous session
            tools[tool_count].load_time = 0;
            tool_count++;
        }
    }

    fclose(f);

    // Reset spindle state - wait for LinuxCNC to tell us which tool is loaded
    spindle_tool_idx = -1;
    last_spindle_state = 0;

    // Update file modification time tracking
    struct stat st;
    if (stat(filename, &st) == 0) {
        last_file_mtime = st.st_mtime;
    }

    log_msg("Loaded %d tools from %s", tool_count, filename);
    log_msg("Waiting for tool load command from LinuxCNC (timers reset)");

    // Print loaded tools to screen for user feedback
    if (tool_count > 0) {
        log_msg("=== Tool Database Loaded ===");
        log_msg("%-5s %-7s %-10s %-10s %s", "Tool", "Pocket", "Z-Offset", "Diameter", "Usage (sec)");
        log_msg("%-5s %-7s %-10s %-10s %s", "----", "------", "--------", "--------", "-----------");

        for (int i = 0; i < MAX_TOOLS; i++) {
            if (!tools[i].in_use) continue;

            Tool *t = &tools[i];
            log_msg("T%-3d  P%-5d  Z%-9.4f  D%-9.4f  %.1f sec",
                    t->toolno, t->pocketno, t->z, t->diameter, t->usage_seconds);
        }

        log_msg("============================");
    }
}

// Reload tools from file while preserving usage data (must be called with mutex held)
static void reload_tools_preserve_usage(const char *filename) {
    FILE *f = fopen(filename, "r");
    if (!f) {
        log_msg("WARNING: Cannot reload file %s: %s", filename, strerror(errno));
        return;
    }

    // Save current usage data and load times
    double saved_usage[MAX_TOOLS];
    time_t saved_load_time[MAX_TOOLS];
    int saved_toolno[MAX_TOOLS];
    int saved_count = tool_count;

    for (int i = 0; i < tool_count; i++) {
        saved_toolno[i] = tools[i].toolno;
        saved_usage[i] = tools[i].usage_seconds;
        saved_load_time[i] = tools[i].load_time;
    }
    int saved_spindle_idx = spindle_tool_idx;

    // Load new data from file
    char line[LINE_LEN];
    tool_count = 0;
    memset(tools, 0, sizeof(tools));

    while (fgets(line, sizeof(line), f) && tool_count < MAX_TOOLS) {
        // Skip empty lines and comments
        if (line[0] == '\n' || line[0] == ';' || line[0] == '#') {
            continue;
        }

        Tool t;
        parse_tool_line(line, &t);

        if (t.toolno >= 0) {
            tools[tool_count] = t;
            tools[tool_count].in_use = 1;

            // Restore usage data for this tool if it existed before
            for (int j = 0; j < saved_count; j++) {
                if (saved_toolno[j] == t.toolno) {
                    tools[tool_count].usage_seconds = saved_usage[j];
                    tools[tool_count].load_time = saved_load_time[j];
                    break;
                }
            }

            tool_count++;
        }
    }

    fclose(f);

    // Restore spindle_tool_idx (find new index for same tool number)
    if (saved_spindle_idx >= 0 && saved_spindle_idx < saved_count) {
        int old_toolno = saved_toolno[saved_spindle_idx];
        spindle_tool_idx = -1;
        for (int i = 0; i < tool_count; i++) {
            if (tools[i].toolno == old_toolno) {
                spindle_tool_idx = i;
                break;
            }
        }
        if (spindle_tool_idx < 0) {
            log_msg("WARNING: Loaded tool T%d not found in reloaded database", old_toolno);
        }
    }

    // Update file modification time
    struct stat st;
    if (stat(filename, &st) == 0) {
        last_file_mtime = st.st_mtime;
    }

    log_msg("Reloaded %d tools from %s (usage data preserved)", tool_count, filename);
}

// Check if file has been modified externally (call before saving)
static int check_file_modified(const char *filename) {
    struct stat st;
    if (stat(filename, &st) == 0) {
        if (st.st_mtime > last_file_mtime) {
            return 1;  // File modified
        }
    }
    return 0;  // File not modified or doesn't exist
}

// Save tools to file (thread-safe with atomic write and version tracking)
void save_tools_to_file(const char *filename, const char *comment) {
    Tool snap[MAX_TOOLS];
    int snap_spindle_idx;
    time_t snap_load_time;
    time_t seen_mtime;
    off_t seen_size;

    pthread_mutex_lock(&tools_mutex);

    // CRITICAL FIX: If file was modified externally, reload NOW under the lock
    // This prevents race conditions and data corruption from concurrent modifications
    time_t cur_mtime;
    off_t cur_size;
    if (get_file_version(filename, &cur_mtime, &cur_size) == 0 &&
        (cur_mtime != db_version.mtime || cur_size != db_version.size)) {
        log_msg("Database modified externally (mtime or size changed), reloading...");
        // Reload while holding the lock to prevent torn reads
        reload_tools_preserve_usage(filename);
        // Refresh version after reload
        if (get_file_version(filename, &db_version.mtime, &db_version.size) != 0) {
            db_version.mtime = 0;
            db_version.size = 0;
        }
    }

    // Snapshot the data quickly under lock
    memcpy(snap, tools, sizeof(tools));
    snap_spindle_idx = spindle_tool_idx;
    snap_load_time = (spindle_tool_idx >= 0 && spindle_tool_idx < MAX_TOOLS)
                     ? tools[spindle_tool_idx].load_time : 0;
    // Remember the version we're based on for conflict detection
    seen_mtime = db_version.mtime;
    seen_size = db_version.size;

    pthread_mutex_unlock(&tools_mutex);

    // Write to temp file (atomic rename at end)
    char tmp[600];
    snprintf(tmp, sizeof(tmp), "%s.tmp", filename);
    FILE *f = fopen(tmp, "w");
    if (!f) {
        log_msg("ERROR: Cannot open %s for writing: %s", tmp, strerror(errno));
        return;
    }

    // Write header with current tool state
    fprintf(f, "; LinuxCNC Tool Database\n");
    fprintf(f, "; Generated: %ld\n", (long)time(NULL));

    // Write current tool info for GUI real-time display
    if (snap_spindle_idx >= 0 && snap_spindle_idx < MAX_TOOLS) {
        fprintf(f, "; CURRENT_TOOL=%d\n", snap[snap_spindle_idx].toolno);
        fprintf(f, "; LOAD_TIME=%ld\n", (long)snap_load_time);
    } else {
        fprintf(f, "; CURRENT_TOOL=0\n");
        fprintf(f, "; LOAD_TIME=0\n");
    }
    fprintf(f, ";\n");

    // Write tool lines from snapshot
    for (int i = 0; i < MAX_TOOLS; i++) {
        if (!snap[i].in_use) continue;

        Tool *t = &snap[i];
        fprintf(f, "T%-3d P%-3d", t->toolno, t->pocketno);

        // Only write non-zero values
        if (fabs(t->x) > 0.00001) fprintf(f, " X%.4f", t->x);
        if (fabs(t->y) > 0.00001) fprintf(f, " Y%.4f", t->y);
        if (fabs(t->z) > 0.00001) fprintf(f, " Z%.4f", t->z);
        if (fabs(t->a) > 0.00001) fprintf(f, " A%.4f", t->a);
        if (fabs(t->b) > 0.00001) fprintf(f, " B%.4f", t->b);
        if (fabs(t->c) > 0.00001) fprintf(f, " C%.4f", t->c);
        if (fabs(t->u) > 0.00001) fprintf(f, " U%.4f", t->u);
        if (fabs(t->v) > 0.00001) fprintf(f, " V%.4f", t->v);
        if (fabs(t->w) > 0.00001) fprintf(f, " W%.4f", t->w);
        if (fabs(t->i) > 0.00001) fprintf(f, " I%.4f", t->i);
        if (fabs(t->j) > 0.00001) fprintf(f, " J%.4f", t->j);
        if (fabs(t->q) > 0.00001) fprintf(f, " Q%.4f", t->q);
        if (fabs(t->diameter) > 0.00001) fprintf(f, " D%.4f", t->diameter);
        if (t->usage_seconds > 0.1) fprintf(f, " M%.1f", t->usage_seconds);

        // Write new tracking fields (ALWAYS write for consistency - no conditionals)
        // This ensures full compatibility with db_tool_manager
        fprintf(f, " UID:%s", strlen(t->uid) > 0 ? t->uid : "LEGACY");
        fprintf(f, " TYPE:%s", strlen(t->type) > 0 ? t->type : "UNKNOWN");
        fprintf(f, " STATUS:%s", strlen(t->status) > 0 ? t->status : "ACTIVE");
        fprintf(f, " CREATED:%ld", t->created > 0 ? (long)t->created : (long)time(NULL));

        // Add timestamp comment (thread-safe localtime_r)
        time_t now = time(NULL);
        struct tm tm_buf;
        struct tm *tm_info = localtime_r(&now, &tm_buf);
        char time_str[64];
        if (tm_info) {
            strftime(time_str, sizeof(time_str), "%d%b%y:%H.%M.%S", tm_info);
            fprintf(f, " ; Tool_%d %s\n", t->toolno, time_str);
        } else {
            fprintf(f, " ; Tool_%d\n", t->toolno);
        }
    }

    // Write history comment if provided
    if (comment && strlen(comment) > 0) {
        fprintf(f, "\n; %s\n", comment);
    }

    // Flush and sync before rename
    fflush(f);
    int fd = fileno(f);
    fsync(fd);
    fclose(f);

    // CRITICAL: Check for concurrent modifications before rename (prevents lost updates)
    // If the file changed while we were writing, abort this write
    time_t now_mtime;
    off_t now_size;
    if (get_file_version(filename, &now_mtime, &now_size) == 0 &&
        (now_mtime != seen_mtime || now_size != seen_size)) {
        // Someone else modified the file while we were writing!
        unlink(tmp);  // Delete our temp file
        log_msg("WARNING: Database changed during save (mtime %ld->%ld, size %ld->%ld); discarding write to avoid conflict",
                (long)seen_mtime, (long)now_mtime, (long)seen_size, (long)now_size);
        return;
    }

    // Atomic rename (no other writer can interrupt this if using file locks)
    if (rename(tmp, filename) != 0) {
        log_msg("ERROR: rename(%s -> %s): %s", tmp, filename, strerror(errno));
        return;
    }

    // Update version tracking under lock
    pthread_mutex_lock(&tools_mutex);
    if (get_file_version(filename, &db_version.mtime, &db_version.size) == 0) {
        last_file_mtime = db_version.mtime;  // Keep old var in sync for now
    }
    pthread_mutex_unlock(&tools_mutex);
}

// Send version handshake to LinuxCNC
void send_version(void) {
    printf("%s\n", DB_VERSION);
    fflush(stdout);
}

// Send a single tool to LinuxCNC
void send_tool(Tool *t) {
    printf("T%d P%d", t->toolno, t->pocketno);

    if (fabs(t->x) > 0.00001) printf(" X%.4f", t->x);
    if (fabs(t->y) > 0.00001) printf(" Y%.4f", t->y);
    if (fabs(t->z) > 0.00001) printf(" Z%.4f", t->z);
    if (fabs(t->a) > 0.00001) printf(" A%.4f", t->a);
    if (fabs(t->b) > 0.00001) printf(" B%.4f", t->b);
    if (fabs(t->c) > 0.00001) printf(" C%.4f", t->c);
    if (fabs(t->u) > 0.00001) printf(" U%.4f", t->u);
    if (fabs(t->v) > 0.00001) printf(" V%.4f", t->v);
    if (fabs(t->w) > 0.00001) printf(" W%.4f", t->w);
    if (fabs(t->i) > 0.00001) printf(" I%.4f", t->i);
    if (fabs(t->j) > 0.00001) printf(" J%.4f", t->j);
    if (fabs(t->q) > 0.00001) printf(" Q%.4f", t->q);
    if (fabs(t->diameter) > 0.00001) printf(" D%.4f", t->diameter);

    printf("\n");
    fflush(stdout);
}

// Handle 'g' command - get all tools
void handle_get(void) {
    pthread_mutex_lock(&tools_mutex);

    for (int i = 0; i < MAX_TOOLS; i++) {
        if (tools[i].in_use) {
            send_tool(&tools[i]);
        }
    }

    pthread_mutex_unlock(&tools_mutex);

    printf("FINI (get_cmd)\n");
    fflush(stdout);
}

// Handle 'p' command - put/update tool parameters
void handle_put(char *params) {
    pthread_mutex_lock(&tools_mutex);

    // Parse params to get tool number first
    Tool temp_tool;
    memset(&temp_tool, 0, sizeof(temp_tool));
    parse_tool_line(params, &temp_tool);

    int idx = find_tool_by_number(temp_tool.toolno);
    if (idx >= 0) {
        // MERGE update into existing tool (preserves all unmentioned fields)
        // This is the key fix: we merge field-by-field instead of overwriting
        parse_tool_line_merge(params, &tools[idx]);
        tools[idx].in_use = 1;

        save_tools_to_file(db_file, "Tool updated (handle_put)");
        log_msg("Updated tool T%d (merged update)", tools[idx].toolno);
    } else {
        log_msg("WARNING: Tool T%d not found for update", temp_tool.toolno);
    }

    pthread_mutex_unlock(&tools_mutex);

    printf("FINI (update recvd) p %s\n", params);
    fflush(stdout);
}

// Internal helper: update tool time (caller must hold mutex)
static void update_tool_time_locked(int idx) {
    if (idx < 0 || idx >= MAX_TOOLS || !tools[idx].in_use) return;
    if (tools[idx].load_time == 0) return;

    time_t now = time(NULL);
    double elapsed_sec = difftime(now, tools[idx].load_time);
    tools[idx].usage_seconds += elapsed_sec;
    tools[idx].load_time = now;  // Reset start time for next period

    log_msg("*** T%d usage updated: +%.1f sec (total: %.1f sec)",
            tools[idx].toolno, elapsed_sec, tools[idx].usage_seconds);
}

// Start tool usage timer (thread-safe)
void start_tool_timer(int idx) {
    pthread_mutex_lock(&tools_mutex);
    if (idx >= 0 && idx < MAX_TOOLS && tools[idx].in_use) {
        tools[idx].load_time = time(NULL);
        log_msg("*** TIMER STARTED for T%d (will count cutting time) ***", tools[idx].toolno);
    }
    pthread_mutex_unlock(&tools_mutex);
}

// Update tool usage time (thread-safe)
void update_tool_time(int idx) {
    pthread_mutex_lock(&tools_mutex);
    update_tool_time_locked(idx);
    pthread_mutex_unlock(&tools_mutex);
}

// Stop tool usage timer (thread-safe)
void stop_tool_timer(int idx) {
    pthread_mutex_lock(&tools_mutex);
    if (idx >= 0 && idx < MAX_TOOLS && tools[idx].in_use) {
        update_tool_time_locked(idx);
        tools[idx].load_time = 0;
        log_msg("*** TIMER STOPPED for T%d (final time: %.1f sec) ***",
                tools[idx].toolno, tools[idx].usage_seconds);
    }
    pthread_mutex_unlock(&tools_mutex);
}

// Pause tool usage timer (when spindle stops) (thread-safe)
void pause_tool_timer(int idx) {
    pthread_mutex_lock(&tools_mutex);
    if (idx >= 0 && idx < MAX_TOOLS && tools[idx].in_use && tools[idx].load_time != 0) {
        // Accumulate time up to now, then clear load_time to pause
        update_tool_time_locked(idx);
        tools[idx].load_time = 0;
        log_msg("--- Timer PAUSED for T%d (spindle stopped, accumulated: %.1f sec) ---",
                tools[idx].toolno, tools[idx].usage_seconds);
    }
    pthread_mutex_unlock(&tools_mutex);
}

// Resume tool usage timer (when spindle starts) (thread-safe)
void resume_tool_timer(int idx) {
    pthread_mutex_lock(&tools_mutex);
    if (idx >= 0 && idx < MAX_TOOLS && tools[idx].in_use && tools[idx].load_time == 0) {
        // Restart timer from now
        tools[idx].load_time = time(NULL);
        log_msg("*** Timer RESUMED for T%d (spindle started, counting...) ***", tools[idx].toolno);
    }
    pthread_mutex_unlock(&tools_mutex);
}

// Check if spindle is running via HAL
int check_spindle_running(void) {
    FILE *fp;
    char result[128];
    int spindle_on = 0;

    // If spindle-aware mode is disabled, always return "on"
    if (!spindle_aware) {
        return 1;
    }

    // Try to read spindle state via halcmd
    // Note: spindle.0.on is a common pin, but could be spindle.N.on for multi-spindle
    fp = popen("halcmd getp spindle.0.on 2>/dev/null", "r");
    if (fp == NULL) {
        // HAL not available or error - fall back to always counting
        return 1;
    }

    if (fgets(result, sizeof(result), fp) != NULL) {
        // Parse result - should be "TRUE" or "FALSE"
        if (strstr(result, "TRUE") != NULL || strstr(result, "1") != NULL) {
            spindle_on = 1;
        }
    }

    pclose(fp);
    return spindle_on;
}

// Check if actually cutting (spindle on + program running + FEED move G1/G2/G3 + motion)
int check_cutting_active(void) {
    FILE *fp;
    char result[256];
    int spindle_on = 0;
    int program_running = 0;
    int is_feed_move = 0;
    int in_motion = 0;

    // If spindle-aware mode is disabled, always return "on"
    if (!spindle_aware) {
        return 1;
    }

    // Check 1: Spindle must be on
    fp = popen("halcmd getp spindle.0.on 2>/dev/null", "r");
    if (fp != NULL) {
        if (fgets(result, sizeof(result), fp) != NULL) {
            if (strstr(result, "TRUE") != NULL || strstr(result, "1") != NULL) {
                spindle_on = 1;
            }
        }
        pclose(fp);
    }

    if (!spindle_on) {
        return 0;  // No spindle, no cutting
    }

    // Check 2: Program must be running
    fp = popen("halcmd getp halui.program.is-running 2>/dev/null", "r");
    if (fp != NULL) {
        if (fgets(result, sizeof(result), fp) != NULL) {
            if (strstr(result, "TRUE") != NULL || strstr(result, "1") != NULL) {
                program_running = 1;
            }
        }
        pclose(fp);
    }

    if (!program_running) {
        return 0;  // Program not running, no cutting
    }

    // Check 3: Motion type must be FEED (G1/G2/G3), not TRAVERSE (G0)
    // motion.motion-type values:
    //   0 = traverse (G0 rapid) - DON'T COUNT
    //   1 = linear feed (G1) - COUNT
    //   2 = arc feed (G2/G3) - COUNT
    //   3 = toolchange - DON'T COUNT
    //   4 = probing - DON'T COUNT
    //   5 = index rotary - DON'T COUNT
    fp = popen("halcmd getp motion.motion-type 2>/dev/null", "r");
    if (fp != NULL) {
        if (fgets(result, sizeof(result), fp) != NULL) {
            int motion_type = atoi(result);
            if (motion_type == 1 || motion_type == 2) {
                is_feed_move = 1;  // G1 linear or G2/G3 arc = cutting move
            }
        }
        pclose(fp);
    }

    if (!is_feed_move) {
        return 0;  // Not a feed move (probably G0 rapid), don't count
    }

    // Check 4: Machine must actually be in motion (velocity > threshold)
    // This prevents counting when program is paused mid-move
    fp = popen("halcmd getp motion.current-vel 2>/dev/null", "r");
    if (fp != NULL) {
        if (fgets(result, sizeof(result), fp) != NULL) {
            double velocity = atof(result);
            if (fabs(velocity) > 0.01) {  // Threshold for motion detection
                in_motion = 1;
            }
        }
        pclose(fp);
    }

    if (!in_motion) {
        return 0;  // Not moving, don't count (paused or at start of move)
    }

    // Only count as cutting if ALL conditions are met:
    // - Spindle ON
    // - Program RUNNING
    // - Motion type is FEED (G1/G2/G3)
    // - Actually MOVING
    int cutting = (spindle_on && program_running && is_feed_move && in_motion);

    static int last_cutting_state = -1;
    if (cutting != last_cutting_state) {
        if (cutting) {
            log_msg(">>> CUTTING STARTED (spindle ON + program running + FEED move G1/G2/G3 + in motion)");
        } else {
            log_msg("<<< CUTTING STOPPED (spindle:%d program:%d feed_move:%d motion:%d)",
                    spindle_on, program_running, is_feed_move, in_motion);
        }
        last_cutting_state = cutting;
    }

    return cutting;
}

// Handle 'l' command - load tool into spindle
void handle_load(char *params) {
    Tool temp;
    parse_tool_line(params, &temp);

    pthread_mutex_lock(&tools_mutex);
    int idx = find_tool_by_number(temp.toolno);
    if (idx >= 0) {
        // Update pocket number from LinuxCNC
        tools[idx].pocketno = temp.pocketno;

        // If another tool was loaded, stop its timer
        int prev_idx = spindle_tool_idx;

        spindle_tool_idx = idx;

        // Start timer (unless T0)
        if (temp.toolno != 0) {
            // Don't start timer on tool load - wait for actual cutting
            // Timer will be managed by periodic_task checking cutting conditions
            tools[idx].load_time = 0;  // Ensure timer is not running yet
            log_msg(">>> Loaded T%d P%d to spindle (ready, timer will start when cutting)",
                    temp.toolno, temp.pocketno);
        }

        double usage = tools[idx].usage_seconds;
        double z = tools[idx].z;
        double diameter = tools[idx].diameter;
        pthread_mutex_unlock(&tools_mutex);

        // Stop previous tool timer (if any) - done outside mutex since stop_tool_timer locks internally
        if (prev_idx >= 0 && prev_idx != idx) {
            stop_tool_timer(prev_idx);
        }

        save_tools_to_file(db_file, "Tool loaded to spindle");
        log_msg("    (Z%.4f D%.4f, usage: %.1f sec)", z, diameter, usage);
    } else {
        pthread_mutex_unlock(&tools_mutex);
        log_msg("WARNING: Tool T%d not found for load", temp.toolno);
    }

    printf("FINI (update recvd) l %s\n", params);
    fflush(stdout);
}

// Handle 'u' command - unload tool from spindle
void handle_unload(char *params) {
    Tool temp;
    parse_tool_line(params, &temp);

    pthread_mutex_lock(&tools_mutex);
    // Unload currently loaded tool (trust spindle_tool_idx, not params)
    if (spindle_tool_idx >= 0 && spindle_tool_idx < MAX_TOOLS) {
        Tool *t = &tools[spindle_tool_idx];

        // If tool was cutting, accumulate final time
        if (t->load_time > 0) {
            time_t now = time(NULL);
            double elapsed_sec = difftime(now, t->load_time);
            t->usage_seconds += elapsed_sec;
            t->load_time = 0;
            log_msg("    Final cut time: +%.1f sec", elapsed_sec);
        }

        // Update pocket number from params if provided
        if (temp.pocketno > 0) {
            t->pocketno = temp.pocketno;
        }

        int toolno = t->toolno;
        int pocketno = t->pocketno;
        double final_usage = t->usage_seconds;

        spindle_tool_idx = -1;
        pthread_mutex_unlock(&tools_mutex);

        save_tools_to_file(db_file, "Tool unloaded from spindle");
        log_msg("<<< Unloaded T%d P%d from spindle (total usage: %.1f sec)",
                toolno, pocketno, final_usage);
    } else {
        pthread_mutex_unlock(&tools_mutex);
        log_msg("WARNING: No tool loaded to unload");
    }

    printf("FINI (update recvd) u %s\n", params);
    fflush(stdout);
}

// Parse and execute a command from LinuxCNC
void parse_command(char *line) {
    // Strip whitespace
    while (*line == ' ' || *line == '\t') line++;

    char cmd = line[0];
    char *params = "";

    if (strlen(line) > 2) {
        params = line + 2;  // Skip "g ", "p ", etc.
    }

    switch (cmd) {
        case 'g':
        case 'G':
            handle_get();
            break;
        case 'p':
        case 'P':
            handle_put(params);
            break;
        case 'l':
        case 'L':
            handle_load(params);
            break;
        case 'u':
        case 'U':
            handle_unload(params);
            break;
        default:
            printf("NAK unknown cmd <%c>\n", cmd);
            fflush(stdout);
            log_msg("Unknown command: %c", cmd);
    }
}

// Main command loop (runs in thread)
void *command_loop(void *arg) {
    (void)arg;  // Unused parameter
    char line[LINE_LEN];
    struct pollfd pfd = { .fd = STDIN_FILENO, .events = POLLIN };

    send_version();

    while (atomic_load(&running) && !got_sigterm) {
        // Poll with 200ms timeout for responsive shutdown
        int rv = poll(&pfd, 1, 200);

        if (rv < 0) {
            if (errno == EINTR) continue;  // Interrupted by signal, retry
            log_msg("ERROR: poll() failed: %s", strerror(errno));
            break;
        }

        if (rv == 0) {
            // Timeout - loop again to check shutdown flags
            continue;
        }

        // Data available on stdin
        if (pfd.revents & POLLIN) {
            if (!fgets(line, sizeof(line), stdin)) {
                // EOF or error
                break;
            }

            // Strip newline
            line[strcspn(line, "\n")] = '\0';

            if (strlen(line) == 0) {
                printf("NAK empty_line\n");
                fflush(stdout);
                continue;
            }

            parse_command(line);
        }

        if (pfd.revents & (POLLERR | POLLHUP)) {
            // Error or hangup on stdin
            break;
        }
    }

    log_msg("Command loop ended");
    atomic_store(&running, 0);

    return NULL;
}

// Periodic task to update tool usage time
void *periodic_task(void *arg) {
    (void)arg;  // Unused parameter
    int save_counter = 0;
    int last_cutting_state = 0;
    time_t cutting_start_time = 0;

    while (atomic_load(&running) && !got_sigterm) {
        // Check every second for real-time accuracy
        sleep(1);

        // Copy state quickly under lock
        int local_spindle_idx;
        int local_spindle_aware;
        pthread_mutex_lock(&tools_mutex);
        local_spindle_idx = spindle_tool_idx;
        local_spindle_aware = spindle_aware;
        pthread_mutex_unlock(&tools_mutex);

        if (local_spindle_idx >= 0 && local_spindle_idx < MAX_TOOLS) {
            // Check if actually cutting (spindle + program + feed move + motion) WITHOUT holding lock
            int cutting = local_spindle_aware ? check_cutting_active() : 1;

            // Now apply updates under lock quickly
            pthread_mutex_lock(&tools_mutex);

            // Verify tool is still the same and in use
            if (spindle_tool_idx != local_spindle_idx || !tools[spindle_tool_idx].in_use) {
                pthread_mutex_unlock(&tools_mutex);
                continue;
            }

            if (local_spindle_aware) {
                // Handle state transitions
                int state_changed = 0;
                if (cutting && !last_cutting_state) {
                    // Just started cutting
                    cutting_start_time = time(NULL);
                    tools[spindle_tool_idx].load_time = cutting_start_time;
                    log_msg("*** CUTTING STARTED for T%d (tracking actual cut time) ***",
                            tools[spindle_tool_idx].toolno);
                    state_changed = 1;
                } else if (!cutting && last_cutting_state) {
                    // Just stopped cutting
                    if (cutting_start_time > 0 && tools[spindle_tool_idx].load_time > 0) {
                        time_t now = time(NULL);
                        double elapsed_sec = difftime(now, cutting_start_time);
                        tools[spindle_tool_idx].usage_seconds += elapsed_sec;
                        tools[spindle_tool_idx].load_time = 0;  // Mark as not cutting

                        log_msg("*** CUTTING STOPPED for T%d: +%.1f sec (total: %.1f sec) ***",
                                tools[spindle_tool_idx].toolno, elapsed_sec,
                                tools[spindle_tool_idx].usage_seconds);
                        state_changed = 1;
                    }
                    cutting_start_time = 0;
                }
                // If still cutting, time accumulates until cutting stops

                last_cutting_state = cutting;
                pthread_mutex_unlock(&tools_mutex);

                // Save immediately on state change so GUI updates instantly
                if (state_changed) {
                    save_tools_to_file(db_file, NULL);
                    save_counter = 0;  // Reset counter since we just saved
                } else {
                    // Regular periodic save every 30 seconds
                    save_counter++;
                    if (save_counter >= 30) {
                        save_tools_to_file(db_file, NULL);
                        save_counter = 0;
                    }
                }
                continue;  // Skip the regular save logic below
            } else {
                // Spindle-aware disabled, count all time when tool is loaded
                if (tools[spindle_tool_idx].load_time > 0) {
                    time_t now = time(NULL);
                    double elapsed_sec = difftime(now, tools[spindle_tool_idx].load_time);
                    if (elapsed_sec >= 1.0) {  // Update every second when not spindle-aware
                        tools[spindle_tool_idx].usage_seconds += elapsed_sec;
                        tools[spindle_tool_idx].load_time = now;
                    }
                }
                pthread_mutex_unlock(&tools_mutex);

                // Regular periodic save every 30 seconds
                save_counter++;
                if (save_counter >= 30) {
                    save_tools_to_file(db_file, NULL);
                    save_counter = 0;
                }
            }
        } else if (local_spindle_idx >= 0) {
            pthread_mutex_lock(&tools_mutex);
            if (spindle_tool_idx >= 0) {
                log_msg("ERROR: Invalid spindle_tool_idx=%d, resetting", spindle_tool_idx);
                spindle_tool_idx = -1;
            }
            pthread_mutex_unlock(&tools_mutex);
        }
    }

    return NULL;
}

// Import tool table from existing tool.tbl file
void import_tool_table(const char *tooltable_file) {
    FILE *f = fopen(tooltable_file, "r");
    if (!f) {
        log_msg("ERROR: Cannot open tool table file: %s", tooltable_file);
        return;
    }

    log_msg("Importing tool table from: %s", tooltable_file);

    char line[LINE_LEN];
    int imported = 0;

    while (fgets(line, sizeof(line), f)) {
        // Skip empty lines and comments
        if (line[0] == '\n' || line[0] == ';' || line[0] == '#') {
            continue;
        }

        Tool t;
        parse_tool_line(line, &t);

        // Only import valid tool numbers
        if (t.toolno < 0 || t.toolno > TOOLNO_MAX) {
            continue;
        }

        // Find if tool already exists
        int idx = find_tool_by_number(t.toolno);
        if (idx >= 0) {
            // Update existing tool, preserve usage time
            double saved_usage = tools[idx].usage_seconds;
            tools[idx] = t;
            tools[idx].usage_seconds = saved_usage;
            tools[idx].load_time = 0;
            tools[idx].in_use = 1;
            imported++;
            log_msg("  Imported T%d: Z%.4f D%.4f P%d",
                    t.toolno, t.z, t.diameter, t.pocketno);
        } else if (tool_count < MAX_TOOLS) {
            // Add new tool
            tools[tool_count] = t;
            tools[tool_count].usage_seconds = 0.0;
            tools[tool_count].load_time = 0;
            tools[tool_count].in_use = 1;
            tool_count++;
            imported++;
            log_msg("  Imported T%d: Z%.4f D%.4f P%d",
                    t.toolno, t.z, t.diameter, t.pocketno);
        }
    }

    fclose(f);

    log_msg("Imported %d tools from tool table", imported);

    // Save to database
    save_tools_to_file(db_file, "Imported from tool table");

    // Print the imported tools
    if (imported > 0) {
        log_msg("=== Tools After Import ===");
        log_msg("%-5s %-7s %-10s %-10s %s", "Tool", "Pocket", "Z-Offset", "Diameter", "Usage (sec)");
        log_msg("%-5s %-7s %-10s %-10s %s", "----", "------", "--------", "--------", "-----------");

        for (int i = 0; i < MAX_TOOLS; i++) {
            if (!tools[i].in_use) continue;

            Tool *t = &tools[i];
            log_msg("T%-3d  P%-5d  Z%-9.4f  D%-9.4f  %.1f sec",
                    t->toolno, t->pocketno, t->z, t->diameter, t->usage_seconds);
        }

        log_msg("==========================");
    }
}

// Signal handler for clean shutdown (async-signal-safe)
void signal_handler(int signo) {
    (void)signo;  // Unused parameter
    got_sigterm = 1;  // Set flag only - this is async-signal-safe
    atomic_store(&running, 0);
}

void print_help(const char *progname) {
    fprintf(stderr, "Usage: %s [options] <database_file>\n", progname);
    fprintf(stderr, "Options:\n");
    fprintf(stderr, "  --period <minutes>          Update period in minutes (default: 1.0)\n");
    fprintf(stderr, "  --random                    Random toolchanger mode\n");
    fprintf(stderr, "  --nonrandom                 Non-random toolchanger mode (default)\n");
    fprintf(stderr, "  --spindle-aware             Count time only when spindle is running (default)\n");
    fprintf(stderr, "  --no-spindle-aware          Count time whenever tool is loaded (legacy mode)\n");
    fprintf(stderr, "  --launch-gui                Launch GUI tool manager on startup (default)\n");
    fprintf(stderr, "  --no-launch-gui             Don't launch GUI tool manager\n");
    fprintf(stderr, "  --import-tooltable <file>   Import existing tool.tbl file\n");
    fprintf(stderr, "  --help                      Show this help\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "Spindle-Aware Mode:\n");
    fprintf(stderr, "  By default, tool usage time is only counted when the spindle is running.\n");
    fprintf(stderr, "  This ensures only actual cutting time is tracked, not rapid moves or\n");
    fprintf(stderr, "  tool changes. Use --no-spindle-aware to count all time (legacy behavior).\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "Examples:\n");
    fprintf(stderr, "  %s --period 0.5 /tmp/my_tools.dat\n", progname);
    fprintf(stderr, "  %s --no-spindle-aware /tmp/my_tools.dat\n", progname);
    fprintf(stderr, "  %s --import-tooltable tool.tbl /tmp/my_tools.dat\n", progname);
    fprintf(stderr, "\n");
    fprintf(stderr, "Import tool table to migrate from TOOL_TABLE to DB_PROGRAM:\n");
    fprintf(stderr, "  %s --import-tooltable ~/linuxcnc/configs/my_machine/tool.tbl \\\n", progname);
    fprintf(stderr, "     ~/linuxcnc_tool_data/my_machine_tools.dat\n");
}

// Launch GUI manager in background
void launch_gui_manager(void) {
    pid_t pid;

    // Check if GUI manager binary exists
    if (access("/usr/local/bin/db_tool_manager", X_OK) != 0) {
        // Try local directory
        if (access("./db_tool_manager", X_OK) != 0) {
            log_msg("Warning: db_tool_manager not found, GUI not launched");
            return;
        }
    }

    // Check if already running to avoid duplicates
    // Use pidof which doesn't match itself like pgrep -f does
    if (system("pidof db_tool_manager > /dev/null 2>&1") == 0) {
        log_msg("GUI manager already running");
        return;
    }

    // Fork to launch GUI in background
    pid = fork();

    if (pid < 0) {
        log_msg("Error: Failed to fork GUI manager: %s", strerror(errno));
        return;
    }

    if (pid == 0) {
        // Child process - launch GUI
        // Close stdin/stdout to detach from parent
        close(STDIN_FILENO);
        close(STDOUT_FILENO);

        // Try installed location first
        if (access("/usr/local/bin/db_tool_manager", X_OK) == 0) {
            execl("/usr/local/bin/db_tool_manager", "db_tool_manager", db_file, (char *)NULL);
        } else {
            // Try local directory
            execl("./db_tool_manager", "db_tool_manager", db_file, (char *)NULL);
        }

        // If exec failed, exit child
        exit(1);
    }

    // Parent process - save PID
    gui_pid = pid;
    log_msg("Launched GUI manager (PID %d) for database: %s", pid, db_file);
}

int main(int argc, char *argv[]) {
    pthread_t periodic_thread;
    char import_file[512] = "";

    // Use stderr for logging
    logfile = stderr;

    // Parse command line arguments
    for (int i = 1; i < argc; i++) {
        if (strcmp(argv[i], "--help") == 0) {
            print_help(argv[0]);
            return 0;
        } else if (strcmp(argv[i], "--period") == 0) {
            if (i + 1 < argc) {
                period_minutes = atof(argv[++i]);
                if (period_minutes <= 0) period_minutes = 1.0;
            }
        } else if (strcmp(argv[i], "--random") == 0) {
            random_toolchanger = 1;
        } else if (strcmp(argv[i], "--nonrandom") == 0) {
            random_toolchanger = 0;
        } else if (strcmp(argv[i], "--spindle-aware") == 0) {
            spindle_aware = 1;
        } else if (strcmp(argv[i], "--no-spindle-aware") == 0) {
            spindle_aware = 0;
        } else if (strcmp(argv[i], "--launch-gui") == 0) {
            launch_gui = 1;
        } else if (strcmp(argv[i], "--no-launch-gui") == 0) {
            launch_gui = 0;
        } else if (strcmp(argv[i], "--import-tooltable") == 0) {
            if (i + 1 < argc) {
                strncpy(import_file, argv[++i], sizeof(import_file) - 1);
                import_file[sizeof(import_file) - 1] = '\0';
            }
        } else {
            // Assume it's the database filename
            strncpy(db_file, argv[i], sizeof(db_file) - 1);
            db_file[sizeof(db_file) - 1] = '\0';
        }
    }

    log_msg("Starting db_tool");
    log_msg("Database file: %s", db_file);
    log_msg("Update period: %.2f minutes", period_minutes);
    log_msg("Toolchanger type: %s", random_toolchanger ? "RANDOM" : "NON-RANDOM");
    log_msg("Spindle-aware mode: %s", spindle_aware ? "ENABLED (counts only cutting time)" : "DISABLED (counts all time)");

    // Set up signal handlers
    signal(SIGINT, signal_handler);
    signal(SIGTERM, signal_handler);
    signal(SIGPIPE, SIG_IGN);  // Ignore SIGPIPE

    // Load tools from file (or create new database)
    load_tools_from_file(db_file);

    // Import tool table if requested
    if (strlen(import_file) > 0) {
        log_msg("Importing tool table: %s", import_file);
        import_tool_table(import_file);
        log_msg("Import complete. Exiting.");
        return 0;  // Exit after import, don't start command loop
    }

    // Launch GUI manager if enabled
    if (launch_gui) {
        launch_gui_manager();
    }

    // Start command loop in main thread (must be main thread for stdin)
    pthread_create(&periodic_thread, NULL, periodic_task, NULL);

    // Run command loop in main thread
    command_loop(NULL);

    // Clean shutdown - wait for periodic thread
    atomic_store(&running, 0);
    pthread_join(periodic_thread, NULL);

    // Graceful shutdown: finalize loaded tool and save
    pthread_mutex_lock(&tools_mutex);
    if (spindle_tool_idx >= 0 && spindle_tool_idx < MAX_TOOLS) {
        Tool *t = &tools[spindle_tool_idx];
        if (t->load_time > 0) {
            time_t now = time(NULL);
            t->usage_seconds += difftime(now, t->load_time);
            t->load_time = 0;
        }
        log_msg("Finalized tool T%d on shutdown", t->toolno);
    }
    pthread_mutex_unlock(&tools_mutex);

    save_tools_to_file(db_file, "Shutdown");
    log_msg("Exiting cleanly");

    return 0;
}
