/*
 * LinuxCNC Tool Database Manager - Unified GTK3 GUI in C
 * Combined real-time monitoring and full editing capabilities
 *
 * Compile: gcc -o db_tool_manager db_tool_manager.c `pkg-config --cflags --libs gtk+-3.0` -lm -pthread
 * Usage: db_tool_manager [database_file]
 */

#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/stat.h>
#include <unistd.h>
#include <math.h>
#include <signal.h>

#define MAX_TOOLS 200
#define LINE_LEN 512

// Column indices for GtkListStore
enum {
    COL_TOOL_NO = 0,
    COL_POCKET,
    COL_X,
    COL_Y,
    COL_Z,
    COL_A,
    COL_B,
    COL_C,
    COL_U,
    COL_V,
    COL_W,
    COL_DIAMETER,
    COL_FRONTANGLE,
    COL_BACKANGLE,
    COL_ORIENTATION,
    COL_USAGE,
    COL_TYPE,
    COL_UID,
    COL_STATUS,
    COL_COMMENT,
    COL_IS_CURRENT,
    NUM_COLS
};

typedef struct {
    int tool_no;
    int pocket;
    double x, y, z;
    double a, b, c;
    double u, v, w;
    double diameter;
    double frontangle;  // I
    double backangle;   // J
    int orientation;    // Q
    double usage_seconds;
    char comment[128];
    int is_current;
    time_t load_time;

    // Tool tracking fields
    char uid[64];           // Unique instance ID
    char type[64];          // Tool type (ENDMILL_6MM, DRILL_3MM, etc.)
    char status[16];        // ACTIVE or RETIRED
    time_t created;         // When added to system
    time_t retired;         // When retired (0 if active)
    char retire_reason[32]; // Worn, Broken, Lost, Superseded
    char retire_notes[256]; // User notes on retirement
} Tool;

typedef struct {
    GtkWidget *window;
    GtkWidget *tree_view;
    GtkListStore *store;
    GtkListStore *type_store;  // Dropdown model for tool types
    GtkWidget *current_tool_label;
    GtkWidget *running_label;
    GtkWidget *update_label;
    GtkWidget *status_label;
    GtkWidget *save_button;
    GtkWidget *reload_button;
    GtkWidget *stats_label;

    char db_file[512];
    int current_tool;
    int prev_current_tool;
    time_t current_tool_load_time;
    time_t last_mtime;
    off_t last_size;  // Track file size for more robust change detection

    Tool tools[MAX_TOOLS];
    int tool_count;
    int updating;
    int modified;
    int auto_reload;  // Auto-reload every second
    int machine_running;  // LinuxCNC running status
    int editing;  // Flag to prevent updates during cell editing
} ManagerApp;

// Common tool types for dropdown
static const char *TOOL_TYPES[] = {
    "ENDMILL",
    "DRILL",
    "FACE_MILL",
    "TAP",
    "REAMER",
    "BORING_BAR",
    "CHAMFER",
    "SLOT_DRILL",
    "BALL_NOSE",
    "CUSTOM",
    NULL
};

// Function prototypes
static gboolean update_data(gpointer data);
static void on_refresh_clicked(GtkWidget *widget, gpointer data);
static void on_save_clicked(GtkWidget *widget, gpointer data);
static void on_reload_clicked(GtkWidget *widget, gpointer data);
static void on_add_tool_clicked(GtkWidget *widget, gpointer data);
static void on_delete_tool_clicked(GtkWidget *widget, gpointer data);
static void on_retire_tool_clicked(GtkWidget *widget, gpointer data);
static void on_generate_uids_clicked(GtkWidget *widget, gpointer data);
static void on_history_clicked(GtkWidget *widget, gpointer data);
static void on_stats_clicked(GtkWidget *widget, gpointer data);
static void on_cell_edited(GtkCellRendererText *renderer, gchar *path, gchar *new_text, gpointer data);
static void on_editing_started(GtkCellRenderer *renderer, GtkCellEditable *editable, gchar *path, gpointer data);
static void on_editing_canceled(GtkCellRenderer *renderer, gpointer data);
static int parse_database(ManagerApp *app);
static void update_display(ManagerApp *app);
static void save_database(ManagerApp *app);
static void reload_to_linuxcnc(ManagerApp *app);
static void create_ui(ManagerApp *app);
static int find_active_database(char *db_path, size_t path_size);
static void set_modified(ManagerApp *app, int modified);
static int check_machine_running(void);
static void text_cell_data_func(GtkTreeViewColumn *col, GtkCellRenderer *renderer,
                                GtkTreeModel *model, GtkTreeIter *iter, gpointer user_data);
static void number_cell_data_func(GtkTreeViewColumn *col, GtkCellRenderer *renderer,
                                  GtkTreeModel *model, GtkTreeIter *iter, gpointer user_data);
static void generate_uid(char *uid, size_t size);
static void retire_tool_to_history(ManagerApp *app, Tool *tool);
static void add_type_to_dropdown(ManagerApp *app, const char *type);

// Add custom type to dropdown if not already present
static void add_type_to_dropdown(ManagerApp *app, const char *type) {
    if (!type || strlen(type) == 0) return;
    if (strcmp(type, "UNKNOWN") == 0 || strcmp(type, "LEGACY") == 0) return;

    GtkTreeIter iter;
    gboolean valid = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(app->type_store), &iter);

    // Check if type already exists in dropdown
    while (valid) {
        gchar *existing_type;
        gtk_tree_model_get(GTK_TREE_MODEL(app->type_store), &iter, 0, &existing_type, -1);

        if (existing_type && strcmp(existing_type, type) == 0) {
            g_free(existing_type);
            return;  // Already exists
        }
        g_free(existing_type);
        valid = gtk_tree_model_iter_next(GTK_TREE_MODEL(app->type_store), &iter);
    }

    // Type doesn't exist, add it to dropdown
    gtk_list_store_append(app->type_store, &iter);
    gtk_list_store_set(app->type_store, &iter, 0, type, -1);
}

// Generate unique tool instance ID
static void generate_uid(char *uid, size_t size) {
    time_t now = time(NULL);
    static int sequence = 0;

    // Format: YYYYMMDDHHMMSS-NNN (e.g., 20251030081530-001)
    struct tm tm_buf;
    struct tm *tm_info = localtime_r(&now, &tm_buf);

    if (tm_info) {
        snprintf(uid, size, "%04d%02d%02d%02d%02d%02d-%03d",
                 tm_info->tm_year + 1900,
                 tm_info->tm_mon + 1,
                 tm_info->tm_mday,
                 tm_info->tm_hour,
                 tm_info->tm_min,
                 tm_info->tm_sec,
                 (sequence++ % 1000));
    } else {
        snprintf(uid, size, "%ld-%03d", (long)now, (sequence++ % 1000));
    }
}

// Mark database as modified
static void set_modified(ManagerApp *app, int modified) {
    app->modified = modified;

    if (modified) {
        char title[600];
        snprintf(title, sizeof(title), "LinuxCNC Tool Manager - %s *MODIFIED*",
                 strrchr(app->db_file, '/') ? strrchr(app->db_file, '/') + 1 : app->db_file);
        gtk_window_set_title(GTK_WINDOW(app->window), title);
        gtk_widget_set_sensitive(app->save_button, TRUE);
    } else {
        char title[600];
        snprintf(title, sizeof(title), "LinuxCNC Tool Manager - %s",
                 strrchr(app->db_file, '/') ? strrchr(app->db_file, '/') + 1 : app->db_file);
        gtk_window_set_title(GTK_WINDOW(app->window), title);
        gtk_widget_set_sensitive(app->save_button, FALSE);
    }
}

// Check if LinuxCNC (db_tool) is running
static int check_machine_running(void) {
    FILE *fp;
    char result[128];
    int running = 0;

    fp = popen("pgrep -x db_tool", "r");
    if (fp == NULL) {
        return 0;
    }

    if (fgets(result, sizeof(result), fp) != NULL) {
        running = 1;
    }

    pclose(fp);
    return running;
}

// Detect actual current tool from LinuxCNC instance
// Returns tool number (0 = no tool loaded, -1 = couldn't detect)
static int detect_current_tool_from_linuxcnc(time_t *load_time) {
    FILE *f;
    char line[LINE_LEN];
    char *home = getenv("HOME");
    char log_path[512];
    int current_tool = 0;  // Default: no tool loaded
    struct stat st;

    *load_time = 0;

    if (!home) return -1;

    // Check LinuxCNC log file for most recent tool load/unload
    snprintf(log_path, sizeof(log_path), "%s/.linuxcnc_print.txt", home);

    f = fopen(log_path, "r");
    if (!f) return -1;

    // Get file modification time (approximate time of last log entry)
    if (fstat(fileno(f), &st) == 0) {
        *load_time = st.st_mtime;
    }

    // Read last 100 lines looking for tool changes
    char lines[100][LINE_LEN];
    int line_count = 0;

    while (fgets(line, sizeof(line), f)) {
        strncpy(lines[line_count % 100], line, LINE_LEN - 1);
        lines[line_count % 100][LINE_LEN - 1] = '\0';
        line_count++;
    }
    fclose(f);

    // Search backwards for most recent load/unload
    int start = (line_count > 100) ? 100 : line_count;
    for (int i = start - 1; i >= 0; i--) {
        int idx = (line_count > 100) ? ((line_count - 100 + i) % 100) : i;

        if (strstr(lines[idx], ">>> Loaded T")) {
            sscanf(lines[idx], "%*[^T]T%d", &current_tool);
            break;
        } else if (strstr(lines[idx], "<<< Unloaded T")) {
            current_tool = 0;  // No tool loaded
            *load_time = 0;
            break;
        }
    }

    return current_tool;
}

// Parse the database file - supports all LinuxCNC fields
static int parse_database(ManagerApp *app) {
    FILE *f;
    char line[LINE_LEN];
    char temp[LINE_LEN];
    int count = 0;

    f = fopen(app->db_file, "r");
    if (!f) {
        return 0;
    }

    // Reset current tool info
    app->current_tool = 0;
    app->current_tool_load_time = 0;

    while (fgets(line, sizeof(line), f) && count < MAX_TOOLS) {
        // Parse header comments for current tool info
        if (line[0] == ';') {
            if (strstr(line, "; CURRENT_TOOL=") == line) {
                sscanf(line, "; CURRENT_TOOL=%d", &app->current_tool);
            } else if (strstr(line, "; LOAD_TIME=") == line) {
                long load_time_long = 0;
                sscanf(line, "; LOAD_TIME=%ld", &load_time_long);
                app->current_tool_load_time = (time_t)load_time_long;
            }
            continue;
        }

        if (line[0] == '\n' || line[0] == '\r') {
            continue;
        }

        Tool *t = &app->tools[count];
        memset(t, 0, sizeof(Tool));

        // Set defaults for new fields
        strcpy(t->status, "ACTIVE");
        strcpy(t->type, "UNKNOWN");
        strcpy(t->uid, "LEGACY");

        // Extract comment first
        char *comment_start = strchr(line, ';');
        if (comment_start) {
            comment_start++;
            while (*comment_start == ' ') comment_start++;
            strncpy(t->comment, comment_start, sizeof(t->comment) - 1);
            t->comment[sizeof(t->comment) - 1] = '\0';
            char *nl = strchr(t->comment, '\n');
            if (nl) *nl = '\0';
            nl = strchr(t->comment, '\r');
            if (nl) *nl = '\0';
        }

        // Parse all fields
        strncpy(temp, line, sizeof(temp) - 1);
        temp[sizeof(temp) - 1] = '\0';
        comment_start = strchr(temp, ';');
        if (comment_start) *comment_start = '\0';

        char *token = strtok(temp, " \t\n");
        while (token != NULL) {
            char letter = token[0];
            double value = 0.0;
            int int_value = 0;

            // Check for new field formats: UID:xxx, TYPE:xxx, STATUS:xxx, CREATED:xxx
            if (strncmp(token, "UID:", 4) == 0) {
                strncpy(t->uid, token + 4, sizeof(t->uid) - 1);
                t->uid[sizeof(t->uid) - 1] = '\0';
            } else if (strncmp(token, "TYPE:", 5) == 0) {
                strncpy(t->type, token + 5, sizeof(t->type) - 1);
                t->type[sizeof(t->type) - 1] = '\0';
            } else if (strncmp(token, "STATUS:", 7) == 0) {
                strncpy(t->status, token + 7, sizeof(t->status) - 1);
                t->status[sizeof(t->status) - 1] = '\0';
            } else if (strncmp(token, "CREATED:", 8) == 0) {
                t->created = (time_t)atol(token + 8);
            } else if (strlen(token) > 1) {
                // Traditional single-letter fields
                if (letter == 'T' || letter == 'P' || letter == 'Q') {
                    int_value = atoi(token + 1);
                } else {
                    value = atof(token + 1);
                }

                switch (letter) {
                    case 'T': case 't': t->tool_no = int_value; break;
                    case 'P': case 'p': t->pocket = int_value; break;
                    case 'X': case 'x': t->x = value; break;
                    case 'Y': case 'y': t->y = value; break;
                    case 'Z': case 'z': t->z = value; break;
                    case 'A': case 'a': t->a = value; break;
                    case 'B': case 'b': t->b = value; break;
                    case 'C': case 'c': t->c = value; break;
                    case 'U': case 'u': t->u = value; break;
                    case 'V': case 'v': t->v = value; break;
                    case 'W': case 'w': t->w = value; break;
                    case 'D': case 'd': t->diameter = value; break;
                    case 'I': case 'i': t->frontangle = value; break;
                    case 'J': case 'j': t->backangle = value; break;
                    case 'Q': case 'q': t->orientation = int_value; break;
                    case 'M': case 'm': t->usage_seconds = value; break;
                }
            }

            token = strtok(NULL, " \t\n");
        }

        if (t->tool_no >= 0) {
            count++;
        }
    }

    fclose(f);
    app->tool_count = count;
    return count;
}

// Save database to file (uses atomic write pattern to avoid TOCTOU)
static void save_database(ManagerApp *app) {
    FILE *f;
    char backup[600];
    char tmpfile[600];
    time_t now = time(NULL);
    struct tm tm_buf;
    struct tm *tm_info;
    char time_str[64];

    // CRITICAL FIX: Write to temp file FIRST (don't remove original yet)
    // This prevents the TOCTOU gap where the DB file doesn't exist
    snprintf(tmpfile, sizeof(tmpfile), "%s.tmp", app->db_file);
    f = fopen(tmpfile, "w");
    if (!f) {
        char msg[600];
        snprintf(msg, sizeof(msg), "Error: Cannot write to %s", tmpfile);
        GtkWidget *dialog = gtk_message_dialog_new(GTK_WINDOW(app->window),
            GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, "%s", msg);
        gtk_dialog_run(GTK_DIALOG(dialog));
        gtk_widget_destroy(dialog);
        return;
    }

    // Write header
    fprintf(f, "; LinuxCNC Tool Database\n");
    fprintf(f, "; Generated: %ld\n", (long)now);
    fprintf(f, "; CURRENT_TOOL=%d\n", app->current_tool);
    fprintf(f, "; LOAD_TIME=%ld\n", (long)app->current_tool_load_time);
    fprintf(f, ";\n");

    // Write all tools (only ACTIVE tools go in main database)
    for (int i = 0; i < app->tool_count; i++) {
        Tool *t = &app->tools[i];

        // Skip retired tools (they belong in history file)
        if (strcmp(t->status, "RETIRED") == 0) {
            continue;
        }

        fprintf(f, "T%-3d P%-3d", t->tool_no, t->pocket);

        // Write all non-zero values
        if (fabs(t->x) > 0.00001) fprintf(f, " X%.4f", t->x);
        if (fabs(t->y) > 0.00001) fprintf(f, " Y%.4f", t->y);
        if (fabs(t->z) > 0.00001) fprintf(f, " Z%.4f", t->z);
        if (fabs(t->a) > 0.00001) fprintf(f, " A%.4f", t->a);
        if (fabs(t->b) > 0.00001) fprintf(f, " B%.4f", t->b);
        if (fabs(t->c) > 0.00001) fprintf(f, " C%.4f", t->c);
        if (fabs(t->u) > 0.00001) fprintf(f, " U%.4f", t->u);
        if (fabs(t->v) > 0.00001) fprintf(f, " V%.4f", t->v);
        if (fabs(t->w) > 0.00001) fprintf(f, " W%.4f", t->w);
        if (fabs(t->diameter) > 0.00001) fprintf(f, " D%.4f", t->diameter);
        if (fabs(t->frontangle) > 0.00001) fprintf(f, " I%.4f", t->frontangle);
        if (fabs(t->backangle) > 0.00001) fprintf(f, " J%.4f", t->backangle);
        if (t->orientation != 0) fprintf(f, " Q%d", t->orientation);
        if (t->usage_seconds > 0.1) fprintf(f, " M%.1f", t->usage_seconds);

        // Write new tracking fields (always write to ensure persistence)
        if (strlen(t->uid) > 0) {
            fprintf(f, " UID:%s", t->uid);
        }
        if (strlen(t->type) > 0) {
            fprintf(f, " TYPE:%s", t->type);
        }
        if (strlen(t->status) > 0) {
            fprintf(f, " STATUS:%s", t->status);
        }
        if (t->created > 0) {
            fprintf(f, " CREATED:%ld", (long)t->created);
        }

        // Add comment (use thread-safe localtime_r)
        if (strlen(t->comment) > 0) {
            fprintf(f, " ; %s\n", t->comment);
        } else {
            tm_info = localtime_r(&now, &tm_buf);
            if (tm_info) {
                strftime(time_str, sizeof(time_str), "%d%b%y:%H.%M.%S", tm_info);
                fprintf(f, " ; Tool_%d %s\n", t->tool_no, time_str);
            } else {
                fprintf(f, " ; Tool_%d\n", t->tool_no);
            }
        }
    }

    // Flush and sync before rename (atomic write)
    fflush(f);
    int fd = fileno(f);
    fsync(fd);
    fclose(f);

    // Atomic rename (this is the critical moment - original still exists until rename completes)
    if (rename(tmpfile, app->db_file) != 0) {
        char msg[600];
        snprintf(msg, sizeof(msg), "Error: Cannot rename temp file to %s", app->db_file);
        GtkWidget *dialog = gtk_message_dialog_new(GTK_WINDOW(app->window),
            GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, "%s", msg);
        gtk_dialog_run(GTK_DIALOG(dialog));
        gtk_widget_destroy(dialog);
        unlink(tmpfile);  // Clean up temp file
        return;
    }

    // Success! Now optionally create backup (after successful write)
    // Note: This is best-effort; if it fails, we still succeeded in saving
    snprintf(backup, sizeof(backup), "%s.backup", app->db_file);
    // Copy the newly written file as backup (ignore errors - backup is optional)
    FILE *src = fopen(app->db_file, "r");
    if (src) {
        FILE *dst = fopen(backup, "w");
        if (dst) {
            char buf[4096];
            size_t n;
            while ((n = fread(buf, 1, sizeof(buf), src)) > 0) {
                fwrite(buf, 1, n, dst);
            }
            fclose(dst);
        }
        fclose(src);
    }

    // Format time for status (recalculate with localtime_r for safety)
    tm_info = localtime_r(&now, &tm_buf);
    if (tm_info) {
        strftime(time_str, sizeof(time_str), "%H:%M:%S", tm_info);
    } else {
        snprintf(time_str, sizeof(time_str), "now");
    }

    char status[256];
    snprintf(status, sizeof(status), "Saved successfully at %s", time_str);
    gtk_label_set_text(GTK_LABEL(app->status_label), status);
    set_modified(app, 0);

    // Update file modification time and size tracking
    struct stat st;
    if (stat(app->db_file, &st) == 0) {
        app->last_mtime = st.st_mtime;
        app->last_size = st.st_size;
    }
}

// Check if UID already exists in history file
static int uid_exists_in_history(const char *history_file, const char *uid) {
    FILE *f;
    char line[LINE_LEN];
    char search_pattern[128];

    if (strlen(uid) == 0 || strcmp(uid, "LEGACY") == 0) {
        return 0;  // Don't check LEGACY UIDs (they're not unique)
    }

    f = fopen(history_file, "r");
    if (!f) {
        return 0;  // File doesn't exist yet, so no duplicates
    }

    snprintf(search_pattern, sizeof(search_pattern), "UID:%s", uid);

    while (fgets(line, sizeof(line), f)) {
        if (strstr(line, search_pattern)) {
            fclose(f);
            return 1;  // Found duplicate
        }
    }

    fclose(f);
    return 0;  // No duplicate found
}

// Retire tool to history file
static void retire_tool_to_history(ManagerApp *app, Tool *tool) {
    char history_file[600];
    FILE *f;
    time_t now = time(NULL);
    struct tm tm_buf;
    struct tm *tm_info;
    char time_str[64];

    snprintf(history_file, sizeof(history_file), "%s.history", app->db_file);

    // Check for duplicate UID in history
    if (uid_exists_in_history(history_file, tool->uid)) {
        char msg[600];
        snprintf(msg, sizeof(msg),
                "Tool UID '%s' already exists in retired history.\n"
                "Skipping history entry (tool will still be removed from active database).",
                tool->uid);
        GtkWidget *dialog = gtk_message_dialog_new(GTK_WINDOW(app->window),
            GTK_DIALOG_MODAL, GTK_MESSAGE_WARNING, GTK_BUTTONS_OK, "%s", msg);
        gtk_dialog_run(GTK_DIALOG(dialog));
        gtk_widget_destroy(dialog);
        return;  // Don't add to history, but allow removal from active DB
    }

    // Append to history file
    f = fopen(history_file, "a");
    if (!f) {
        char msg[600];
        snprintf(msg, sizeof(msg), "Error: Cannot write to history file %s", history_file);
        GtkWidget *dialog = gtk_message_dialog_new(GTK_WINDOW(app->window),
            GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, "%s", msg);
        gtk_dialog_run(GTK_DIALOG(dialog));
        gtk_widget_destroy(dialog);
        return;
    }

    // Write tool entry to history
    fprintf(f, "T%-3d P%-3d", tool->tool_no, tool->pocket);

    if (fabs(tool->x) > 0.00001) fprintf(f, " X%.4f", tool->x);
    if (fabs(tool->y) > 0.00001) fprintf(f, " Y%.4f", tool->y);
    if (fabs(tool->z) > 0.00001) fprintf(f, " Z%.4f", tool->z);
    if (fabs(tool->a) > 0.00001) fprintf(f, " A%.4f", tool->a);
    if (fabs(tool->b) > 0.00001) fprintf(f, " B%.4f", tool->b);
    if (fabs(tool->c) > 0.00001) fprintf(f, " C%.4f", tool->c);
    if (fabs(tool->u) > 0.00001) fprintf(f, " U%.4f", tool->u);
    if (fabs(tool->v) > 0.00001) fprintf(f, " V%.4f", tool->v);
    if (fabs(tool->w) > 0.00001) fprintf(f, " W%.4f", tool->w);
    if (fabs(tool->diameter) > 0.00001) fprintf(f, " D%.4f", tool->diameter);
    if (fabs(tool->frontangle) > 0.00001) fprintf(f, " I%.4f", tool->frontangle);
    if (fabs(tool->backangle) > 0.00001) fprintf(f, " J%.4f", tool->backangle);
    if (tool->orientation != 0) fprintf(f, " Q%d", tool->orientation);
    if (tool->usage_seconds > 0.1) fprintf(f, " M%.1f", tool->usage_seconds);

    // Write tracking fields
    if (strlen(tool->uid) > 0) fprintf(f, " UID:%s", tool->uid);
    if (strlen(tool->type) > 0) fprintf(f, " TYPE:%s", tool->type);
    fprintf(f, " STATUS:RETIRED");
    if (tool->created > 0) fprintf(f, " CREATED:%ld", (long)tool->created);
    fprintf(f, " RETIRED:%ld", (long)tool->retired);

    // Write retire reason (replace spaces with underscores for parsing)
    if (strlen(tool->retire_reason) > 0) {
        char reason_escaped[256];
        strncpy(reason_escaped, tool->retire_reason, sizeof(reason_escaped) - 1);
        reason_escaped[sizeof(reason_escaped) - 1] = '\0';
        for (char *p = reason_escaped; *p; p++) {
            if (*p == ' ') *p = '_';
        }
        fprintf(f, " RETIRE_REASON:%s", reason_escaped);
    }

    // Write retire notes (replace spaces with underscores for parsing)
    if (strlen(tool->retire_notes) > 0) {
        char notes_escaped[256];
        strncpy(notes_escaped, tool->retire_notes, sizeof(notes_escaped) - 1);
        notes_escaped[sizeof(notes_escaped) - 1] = '\0';
        for (char *p = notes_escaped; *p; p++) {
            if (*p == ' ') *p = '_';
        }
        fprintf(f, " RETIRE_NOTES:%s", notes_escaped);
    }

    // Write comment with retirement info
    tm_info = localtime_r(&now, &tm_buf);
    if (tm_info) {
        strftime(time_str, sizeof(time_str), "%Y-%m-%d %H:%M:%S", tm_info);
    } else {
        snprintf(time_str, sizeof(time_str), "now");
    }

    fprintf(f, " ; RETIRED %s - %s", time_str, tool->retire_reason);
    if (strlen(tool->retire_notes) > 0) {
        fprintf(f, " - %s", tool->retire_notes);
    }
    if (strlen(tool->comment) > 0) {
        fprintf(f, " [%s]", tool->comment);
    }
    fprintf(f, "\n");

    fclose(f);
}

// Reload to LinuxCNC
static void reload_to_linuxcnc(ManagerApp *app) {
    FILE *fp;
    char line[256];
    int pid_found = 0;

    fp = popen("pgrep -x db_tool", "r");
    if (fp) {
        if (fgets(line, sizeof(line), fp)) {
            int pid = atoi(line);
            if (pid > 0) {
                pid_found = 1;
                char msg[512];
                snprintf(msg, sizeof(msg),
                    "LinuxCNC db_tool is running (PID %d).\n"
                    "Database saved - changes will be picked up automatically.", pid);

                GtkWidget *dialog = gtk_message_dialog_new(GTK_WINDOW(app->window),
                    GTK_DIALOG_MODAL, GTK_MESSAGE_INFO, GTK_BUTTONS_OK, "%s", msg);
                gtk_dialog_run(GTK_DIALOG(dialog));
                gtk_widget_destroy(dialog);
            }
        }
        pclose(fp);
    }

    if (!pid_found) {
        GtkWidget *dialog = gtk_message_dialog_new(GTK_WINDOW(app->window),
            GTK_DIALOG_MODAL, GTK_MESSAGE_WARNING, GTK_BUTTONS_OK,
            "Warning: LinuxCNC db_tool is not running.\n"
            "Database saved, but won't be loaded until LinuxCNC starts.");
        gtk_dialog_run(GTK_DIALOG(dialog));
        gtk_widget_destroy(dialog);
    }

    gtk_label_set_text(GTK_LABEL(app->status_label), "Reloaded to LinuxCNC");
}

// Cell highlighting for current tool
static void text_cell_data_func(GtkTreeViewColumn *col, GtkCellRenderer *renderer,
                                GtkTreeModel *model, GtkTreeIter *iter, gpointer user_data) {
    (void)col;
    (void)user_data;
    gboolean is_current;

    gtk_tree_model_get(model, iter, COL_IS_CURRENT, &is_current, -1);

    if (is_current) {
        g_object_set(renderer,
                    "cell-background", "#90EE90",
                    "weight", PANGO_WEIGHT_BOLD,
                    NULL);
    } else {
        g_object_set(renderer,
                    "cell-background", NULL,
                    "weight", PANGO_WEIGHT_NORMAL,
                    NULL);
    }
}

// Number formatting with highlighting
static void number_cell_data_func(GtkTreeViewColumn *col, GtkCellRenderer *renderer,
                                  GtkTreeModel *model, GtkTreeIter *iter, gpointer user_data) {
    (void)col;
    int col_id = GPOINTER_TO_INT(user_data);
    gboolean is_current;
    char text[64];

    gtk_tree_model_get(model, iter, COL_IS_CURRENT, &is_current, -1);

    if (col_id == COL_TOOL_NO || col_id == COL_POCKET || col_id == COL_ORIENTATION) {
        int value;
        gtk_tree_model_get(model, iter, col_id, &value, -1);
        snprintf(text, sizeof(text), "%d", value);
    } else {
        double value;
        gtk_tree_model_get(model, iter, col_id, &value, -1);

        if (col_id == COL_USAGE) {
            snprintf(text, sizeof(text), "%.3f", value);
        } else if (fabs(value) < 0.0001) {
            snprintf(text, sizeof(text), "0");
        } else {
            snprintf(text, sizeof(text), "%.4f", value);
        }
    }

    g_object_set(renderer, "text", text, NULL);

    if (is_current) {
        g_object_set(renderer,
                    "cell-background", "#90EE90",
                    "weight", PANGO_WEIGHT_BOLD,
                    NULL);
    } else {
        g_object_set(renderer,
                    "cell-background", NULL,
                    "weight", PANGO_WEIGHT_NORMAL,
                    NULL);
    }
}

// Update display
static void update_display(ManagerApp *app) {
    GtkTreeIter iter;
    time_t now = time(NULL);
    GtkTreeSelection *selection;
    int selected_tool_no = -1;
    gboolean had_selection = FALSE;
    gint sort_column_id;
    GtkSortType sort_order;
    gboolean was_sorted = FALSE;

    // Skip updates if user is currently editing a cell
    if (app->editing) {
        return;
    }

    app->updating = 1;

    // Save current selection by tool number (not path, which can change due to sorting)
    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(app->tree_view));
    if (gtk_tree_selection_get_selected(selection, NULL, &iter)) {
        gtk_tree_model_get(GTK_TREE_MODEL(app->store), &iter, COL_TOOL_NO, &selected_tool_no, -1);
        had_selection = TRUE;
    }

    // Temporarily disable sorting to prevent jumping during value updates
    if (gtk_tree_sortable_get_sort_column_id(GTK_TREE_SORTABLE(app->store),
                                              &sort_column_id, &sort_order)) {
        was_sorted = TRUE;
        gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(app->store),
                                             GTK_TREE_SORTABLE_UNSORTED_SORT_COLUMN_ID,
                                             GTK_SORT_ASCENDING);
    }

    // Update current tool label
    if (app->current_tool != app->prev_current_tool) {
        char markup[256];
        if (app->current_tool > 0) {
            snprintf(markup, sizeof(markup),
                     "<b>Current Tool:</b> <span foreground='#008000' size='x-large'>T%d</span>",
                     app->current_tool);
        } else {
            snprintf(markup, sizeof(markup), "<b>Current Tool:</b> <span foreground='gray'>None</span>");
        }
        gtk_label_set_markup(GTK_LABEL(app->current_tool_label), markup);
        app->prev_current_tool = app->current_tool;
    }

    // Update running status
    int machine_running = check_machine_running();
    if (machine_running != app->machine_running) {
        char markup[256];
        if (machine_running) {
            snprintf(markup, sizeof(markup), "<b>RUNNING:</b> <span foreground='green' size='large'>ON</span>");
        } else {
            snprintf(markup, sizeof(markup), "<b>RUNNING:</b> <span foreground='red'>OFF</span>");
        }
        gtk_label_set_markup(GTK_LABEL(app->running_label), markup);
        app->machine_running = machine_running;
    }

    // Calculate statistics
    int tools_with_usage = 0;
    double total_usage = 0.0;
    double max_usage = 0.0;
    int max_usage_tool = 0;

    // Check if we need to rebuild list (tool count changed) or just update
    gint list_count = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(app->store), NULL);
    gboolean need_rebuild = (list_count != app->tool_count);

    if (need_rebuild) {
        // Full rebuild only when tool count changes
        gtk_list_store_clear(app->store);
    }

    // Get first row for incremental update
    gboolean valid = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(app->store), &iter);

    for (int i = 0; i < app->tool_count; i++) {
        Tool *t = &app->tools[i];
        double display_usage = t->usage_seconds;
        gboolean is_current = (t->tool_no == app->current_tool);

        // Add live elapsed time for current tool
        if (is_current && t->load_time > 0) {
            double elapsed_sec = difftime(now, t->load_time);
            display_usage += elapsed_sec;
        }

        // Update statistics
        if (display_usage > 0.001) {
            tools_with_usage++;
            total_usage += display_usage;
            if (display_usage > max_usage) {
                max_usage = display_usage;
                max_usage_tool = t->tool_no;
            }
        }

        if (need_rebuild || !valid) {
            // Full rebuild or append new rows
            gtk_list_store_append(app->store, &iter);
            gtk_list_store_set(app->store, &iter,
                              COL_TOOL_NO, t->tool_no,
                              COL_POCKET, t->pocket,
                              COL_X, t->x,
                              COL_Y, t->y,
                              COL_Z, t->z,
                              COL_A, t->a,
                              COL_B, t->b,
                              COL_C, t->c,
                              COL_U, t->u,
                              COL_V, t->v,
                              COL_W, t->w,
                              COL_DIAMETER, t->diameter,
                              COL_FRONTANGLE, t->frontangle,
                              COL_BACKANGLE, t->backangle,
                              COL_ORIENTATION, t->orientation,
                              COL_USAGE, display_usage,
                              COL_TYPE, t->type,
                              COL_UID, t->uid,
                              COL_STATUS, t->status,
                              COL_COMMENT, t->comment,
                              COL_IS_CURRENT, is_current,
                              -1);
        } else {
            // Incremental update - update all columns to reflect any edits
            gtk_list_store_set(app->store, &iter,
                              COL_TOOL_NO, t->tool_no,
                              COL_POCKET, t->pocket,
                              COL_X, t->x,
                              COL_Y, t->y,
                              COL_Z, t->z,
                              COL_A, t->a,
                              COL_B, t->b,
                              COL_C, t->c,
                              COL_U, t->u,
                              COL_V, t->v,
                              COL_W, t->w,
                              COL_DIAMETER, t->diameter,
                              COL_FRONTANGLE, t->frontangle,
                              COL_BACKANGLE, t->backangle,
                              COL_ORIENTATION, t->orientation,
                              COL_USAGE, display_usage,
                              COL_TYPE, t->type,
                              COL_UID, t->uid,
                              COL_STATUS, t->status,
                              COL_COMMENT, t->comment,
                              COL_IS_CURRENT, is_current,
                              -1);
            // Move to next row
            valid = gtk_tree_model_iter_next(GTK_TREE_MODEL(app->store), &iter);
        }
    }

    // Remove any extra rows if list was longer than tool count
    if (!need_rebuild) {
        while (valid) {
            GtkTreeIter next = iter;
            valid = gtk_tree_model_iter_next(GTK_TREE_MODEL(app->store), &next);
            gtk_list_store_remove(app->store, &iter);
            iter = next;
        }
    }

    // Restore selection if it existed - find the tool by number, not by path
    if (had_selection && selected_tool_no >= 0) {
        gboolean valid = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(app->store), &iter);
        while (valid) {
            int tool_no;
            gtk_tree_model_get(GTK_TREE_MODEL(app->store), &iter, COL_TOOL_NO, &tool_no, -1);
            if (tool_no == selected_tool_no) {
                gtk_tree_selection_select_iter(selection, &iter);
                break;
            }
            valid = gtk_tree_model_iter_next(GTK_TREE_MODEL(app->store), &iter);
        }
    }

    // Update statistics label
    char stats[512];
    if (tools_with_usage > 0) {
        snprintf(stats, sizeof(stats),
                 "<b>%d</b> tools | <b>%d</b> used | Total: <b>%.1f</b> sec | Avg: <b>%.1f</b> sec | Max: T%d (<b>%.1f</b> sec)",
                 app->tool_count, tools_with_usage, total_usage,
                 total_usage / tools_with_usage, max_usage_tool, max_usage);
    } else {
        snprintf(stats, sizeof(stats), "<b>%d</b> tools | No usage recorded yet", app->tool_count);
    }
    gtk_label_set_markup(GTK_LABEL(app->stats_label), stats);

    // Update timestamp
    struct tm *tm_info = localtime(&now);
    char time_str[64];
    strftime(time_str, sizeof(time_str), "%H:%M:%S", tm_info);
    char markup[128];
    snprintf(markup, sizeof(markup), "<small>Updated: %s</small>", time_str);
    gtk_label_set_markup(GTK_LABEL(app->update_label), markup);

    // Re-enable sorting
    if (was_sorted) {
        gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(app->store),
                                             sort_column_id, sort_order);
    }

    app->updating = 0;
}

// Editing started callback
static void on_editing_started(GtkCellRenderer *renderer, GtkCellEditable *editable, gchar *path, gpointer data) {
    (void)renderer;
    (void)editable;
    (void)path;
    ManagerApp *app = (ManagerApp *)data;
    app->editing = 1;  // Set flag to prevent updates during editing
}

// Editing canceled callback
static void on_editing_canceled(GtkCellRenderer *renderer, gpointer data) {
    (void)renderer;
    ManagerApp *app = (ManagerApp *)data;
    app->editing = 0;  // Clear editing flag
}

// Cell edited callback
static void on_cell_edited(GtkCellRendererText *renderer, gchar *path_str, gchar *new_text, gpointer data) {
    ManagerApp *app = (ManagerApp *)data;
    GtkTreePath *path = gtk_tree_path_new_from_string(path_str);
    GtkTreeIter iter;
    int row;

    app->editing = 0;  // Clear editing flag

    if (!gtk_tree_model_get_iter(GTK_TREE_MODEL(app->store), &iter, path)) {
        gtk_tree_path_free(path);
        return;
    }

    row = atoi(path_str);
    gtk_tree_path_free(path);

    if (row < 0 || row >= app->tool_count) return;

    Tool *t = &app->tools[row];
    int col = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(renderer), "column"));

    // Update the tool based on column
    switch (col) {
        case COL_TOOL_NO: t->tool_no = atoi(new_text); break;
        case COL_POCKET: t->pocket = atoi(new_text); break;
        case COL_X: t->x = atof(new_text); break;
        case COL_Y: t->y = atof(new_text); break;
        case COL_Z: t->z = atof(new_text); break;
        case COL_A: t->a = atof(new_text); break;
        case COL_B: t->b = atof(new_text); break;
        case COL_C: t->c = atof(new_text); break;
        case COL_U: t->u = atof(new_text); break;
        case COL_V: t->v = atof(new_text); break;
        case COL_W: t->w = atof(new_text); break;
        case COL_DIAMETER: t->diameter = atof(new_text); break;
        case COL_FRONTANGLE: t->frontangle = atof(new_text); break;
        case COL_BACKANGLE: t->backangle = atof(new_text); break;
        case COL_ORIENTATION: t->orientation = atoi(new_text); break;
        case COL_TYPE:
            strncpy(t->type, new_text, sizeof(t->type) - 1);
            t->type[sizeof(t->type) - 1] = '\0';
            // Add custom type to dropdown for future use
            add_type_to_dropdown(app, new_text);
            break;
        case COL_COMMENT:
            strncpy(t->comment, new_text, sizeof(t->comment) - 1);
            t->comment[sizeof(t->comment) - 1] = '\0';
            break;
    }

    set_modified(app, 1);
    update_display(app);
}

// Button callbacks
static void on_save_clicked(GtkWidget *widget, gpointer data) {
    (void)widget;
    save_database((ManagerApp *)data);
}

static void on_reload_clicked(GtkWidget *widget, gpointer data) {
    (void)widget;
    ManagerApp *app = (ManagerApp *)data;

    if (app->modified) {
        GtkWidget *dialog = gtk_message_dialog_new(GTK_WINDOW(app->window),
            GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO,
            "Save changes before reloading to LinuxCNC?");
        int response = gtk_dialog_run(GTK_DIALOG(dialog));
        gtk_widget_destroy(dialog);

        if (response == GTK_RESPONSE_YES) {
            save_database(app);
        }
    }

    reload_to_linuxcnc(app);
}

static void on_add_tool_clicked(GtkWidget *widget, gpointer data) {
    (void)widget;
    ManagerApp *app = (ManagerApp *)data;

    if (app->tool_count >= MAX_TOOLS) {
        GtkWidget *dialog = gtk_message_dialog_new(GTK_WINDOW(app->window),
            GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
            "Maximum tools (%d) reached", MAX_TOOLS);
        gtk_dialog_run(GTK_DIALOG(dialog));
        gtk_widget_destroy(dialog);
        return;
    }

    // Create dialog for tool type selection
    GtkWidget *dialog = gtk_dialog_new_with_buttons("Add New Tool",
        GTK_WINDOW(app->window),
        GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
        "Add", GTK_RESPONSE_ACCEPT,
        "Cancel", GTK_RESPONSE_CANCEL,
        NULL);

    GtkWidget *content = gtk_dialog_get_content_area(GTK_DIALOG(dialog));
    GtkWidget *vbox = gtk_box_new(GTK_ORIENTATION_VERTICAL, 10);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), 10);
    gtk_container_add(GTK_CONTAINER(content), vbox);

    // Type selection - use shared type_store to show custom types
    GtkWidget *type_label = gtk_label_new("Select Tool Type:");
    gtk_label_set_xalign(GTK_LABEL(type_label), 0);
    gtk_box_pack_start(GTK_BOX(vbox), type_label, FALSE, FALSE, 0);

    GtkWidget *type_combo = gtk_combo_box_new_with_model_and_entry(GTK_TREE_MODEL(app->type_store));
    gtk_combo_box_set_entry_text_column(GTK_COMBO_BOX(type_combo), 0);
    gtk_combo_box_set_active(GTK_COMBO_BOX(type_combo), 0);  // Select first type by default
    gtk_box_pack_start(GTK_BOX(vbox), type_combo, FALSE, FALSE, 0);

    gtk_widget_show_all(dialog);

    int response = gtk_dialog_run(GTK_DIALOG(dialog));

    if (response == GTK_RESPONSE_ACCEPT) {
        // Get selected type from combo box entry
        GtkWidget *entry = gtk_bin_get_child(GTK_BIN(type_combo));
        const gchar *selected_type = gtk_entry_get_text(GTK_ENTRY(entry));

        // Find next available tool number
        int new_tool_no = 1;
        for (int i = 0; i < app->tool_count; i++) {
            if (app->tools[i].tool_no >= new_tool_no) {
                new_tool_no = app->tools[i].tool_no + 1;
            }
        }

        Tool *t = &app->tools[app->tool_count];
        memset(t, 0, sizeof(Tool));
        t->tool_no = new_tool_no;
        t->pocket = new_tool_no;

        // Set new tracking fields
        generate_uid(t->uid, sizeof(t->uid));
        if (selected_type && strlen(selected_type) > 0) {
            strncpy(t->type, selected_type, sizeof(t->type) - 1);
            t->type[sizeof(t->type) - 1] = '\0';
            // Add to dropdown for future use
            add_type_to_dropdown(app, selected_type);
        } else {
            strcpy(t->type, "UNKNOWN");
        }
        strcpy(t->status, "ACTIVE");
        t->created = time(NULL);

        snprintf(t->comment, sizeof(t->comment), "New Tool - %s", t->type);

        app->tool_count++;

        // Auto-save after adding new tool to persist changes immediately
        save_database(app);
        update_display(app);

        gtk_label_set_text(GTK_LABEL(app->status_label), "Tool added and saved");
    }

    gtk_widget_destroy(dialog);
}

static void on_retire_tool_clicked(GtkWidget *widget, gpointer data) {
    (void)widget;
    ManagerApp *app = (ManagerApp *)data;
    GtkTreeSelection *selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(app->tree_view));
    GtkTreeIter iter;
    GtkTreeModel *model;

    if (!gtk_tree_selection_get_selected(selection, &model, &iter)) {
        GtkWidget *dialog = gtk_message_dialog_new(GTK_WINDOW(app->window),
            GTK_DIALOG_MODAL, GTK_MESSAGE_INFO, GTK_BUTTONS_OK,
            "Select a tool to retire");
        gtk_dialog_run(GTK_DIALOG(dialog));
        gtk_widget_destroy(dialog);
        return;
    }

    GtkTreePath *path = gtk_tree_model_get_path(model, &iter);
    int row = gtk_tree_path_get_indices(path)[0];
    gtk_tree_path_free(path);

    if (row < 0 || row >= app->tool_count) return;

    Tool *t = &app->tools[row];

    // Create retirement dialog
    GtkWidget *dialog = gtk_dialog_new_with_buttons("Retire Tool",
        GTK_WINDOW(app->window),
        GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
        "Retire", GTK_RESPONSE_ACCEPT,
        "Cancel", GTK_RESPONSE_CANCEL,
        NULL);

    GtkWidget *content = gtk_dialog_get_content_area(GTK_DIALOG(dialog));
    GtkWidget *vbox = gtk_box_new(GTK_ORIENTATION_VERTICAL, 10);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), 10);
    gtk_container_add(GTK_CONTAINER(content), vbox);

    // Show tool info
    char info[256];
    snprintf(info, sizeof(info), "Retiring: T%d - %s (UID: %s)\nUsage: %.1f seconds",
             t->tool_no, t->type, t->uid, t->usage_seconds);
    GtkWidget *info_label = gtk_label_new(info);
    gtk_label_set_xalign(GTK_LABEL(info_label), 0);
    gtk_box_pack_start(GTK_BOX(vbox), info_label, FALSE, FALSE, 0);

    // Reason selection
    GtkWidget *reason_label = gtk_label_new("Retirement Reason:");
    gtk_label_set_xalign(GTK_LABEL(reason_label), 0);
    gtk_box_pack_start(GTK_BOX(vbox), reason_label, FALSE, FALSE, 0);

    GtkWidget *reason_combo = gtk_combo_box_text_new();
    gtk_combo_box_text_append_text(GTK_COMBO_BOX_TEXT(reason_combo), "Worn");
    gtk_combo_box_text_append_text(GTK_COMBO_BOX_TEXT(reason_combo), "Broken");
    gtk_combo_box_text_append_text(GTK_COMBO_BOX_TEXT(reason_combo), "Lost");
    gtk_combo_box_text_append_text(GTK_COMBO_BOX_TEXT(reason_combo), "Superseded");
    gtk_combo_box_set_active(GTK_COMBO_BOX(reason_combo), 0);
    gtk_box_pack_start(GTK_BOX(vbox), reason_combo, FALSE, FALSE, 0);

    // Notes entry
    GtkWidget *notes_label = gtk_label_new("Notes (optional):");
    gtk_label_set_xalign(GTK_LABEL(notes_label), 0);
    gtk_box_pack_start(GTK_BOX(vbox), notes_label, FALSE, FALSE, 0);

    GtkWidget *notes_entry = gtk_entry_new();
    gtk_entry_set_placeholder_text(GTK_ENTRY(notes_entry), "Enter any additional notes...");
    gtk_box_pack_start(GTK_BOX(vbox), notes_entry, FALSE, FALSE, 0);

    gtk_widget_show_all(dialog);

    int response = gtk_dialog_run(GTK_DIALOG(dialog));

    if (response == GTK_RESPONSE_ACCEPT) {
        gchar *reason = gtk_combo_box_text_get_active_text(GTK_COMBO_BOX_TEXT(reason_combo));
        const gchar *notes = gtk_entry_get_text(GTK_ENTRY(notes_entry));

        // Set retirement fields
        t->retired = time(NULL);
        if (reason) {
            strncpy(t->retire_reason, reason, sizeof(t->retire_reason) - 1);
            t->retire_reason[sizeof(t->retire_reason) - 1] = '\0';
            g_free(reason);
        } else {
            strcpy(t->retire_reason, "Unknown");
        }

        if (notes && strlen(notes) > 0) {
            strncpy(t->retire_notes, notes, sizeof(t->retire_notes) - 1);
            t->retire_notes[sizeof(t->retire_notes) - 1] = '\0';
        }

        strcpy(t->status, "RETIRED");

        // Write to history file
        retire_tool_to_history(app, t);

        // Remove from active tools array
        for (int i = row; i < app->tool_count - 1; i++) {
            app->tools[i] = app->tools[i + 1];
        }
        app->tool_count--;

        // Auto-save after retirement to persist changes immediately
        save_database(app);
        update_display(app);
        gtk_label_set_text(GTK_LABEL(app->status_label), "Tool retired to history and saved");
    }

    gtk_widget_destroy(dialog);
}

static void on_delete_tool_clicked(GtkWidget *widget, gpointer data) {
    (void)widget;
    ManagerApp *app = (ManagerApp *)data;
    GtkTreeSelection *selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(app->tree_view));
    GtkTreeIter iter;
    GtkTreeModel *model;

    if (!gtk_tree_selection_get_selected(selection, &model, &iter)) {
        GtkWidget *dialog = gtk_message_dialog_new(GTK_WINDOW(app->window),
            GTK_DIALOG_MODAL, GTK_MESSAGE_INFO, GTK_BUTTONS_OK,
            "Select a tool to delete");
        gtk_dialog_run(GTK_DIALOG(dialog));
        gtk_widget_destroy(dialog);
        return;
    }

    GtkTreePath *path = gtk_tree_model_get_path(model, &iter);
    int row = gtk_tree_path_get_indices(path)[0];
    gtk_tree_path_free(path);

    if (row < 0 || row >= app->tool_count) return;

    GtkWidget *dialog = gtk_message_dialog_new(GTK_WINDOW(app->window),
        GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO,
        "Delete tool T%d?", app->tools[row].tool_no);
    int response = gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_destroy(dialog);

    if (response == GTK_RESPONSE_YES) {
        for (int i = row; i < app->tool_count - 1; i++) {
            app->tools[i] = app->tools[i + 1];
        }
        app->tool_count--;

        // Auto-save after deletion to persist changes immediately
        save_database(app);
        update_display(app);
        gtk_label_set_text(GTK_LABEL(app->status_label), "Tool deleted and saved");
    }
}

static void on_generate_uids_clicked(GtkWidget *widget, gpointer data) {
    (void)widget;
    ManagerApp *app = (ManagerApp *)data;

    // Count legacy tools
    int legacy_count = 0;
    for (int i = 0; i < app->tool_count; i++) {
        if (strcmp(app->tools[i].uid, "LEGACY") == 0) {
            legacy_count++;
        }
    }

    if (legacy_count == 0) {
        GtkWidget *dialog = gtk_message_dialog_new(GTK_WINDOW(app->window),
            GTK_DIALOG_MODAL, GTK_MESSAGE_INFO, GTK_BUTTONS_OK,
            "No legacy tools found.\n\nAll tools already have unique IDs.");
        gtk_dialog_run(GTK_DIALOG(dialog));
        gtk_widget_destroy(dialog);
        return;
    }

    // Confirm action
    char msg[256];
    snprintf(msg, sizeof(msg),
             "Found %d legacy tool%s without unique IDs.\n\n"
             "Generate unique IDs for %s?",
             legacy_count,
             legacy_count == 1 ? "" : "s",
             legacy_count == 1 ? "this tool" : "these tools");

    GtkWidget *dialog = gtk_message_dialog_new(GTK_WINDOW(app->window),
        GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO, "%s", msg);
    int response = gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_destroy(dialog);

    if (response == GTK_RESPONSE_YES) {
        int generated = 0;

        for (int i = 0; i < app->tool_count; i++) {
            if (strcmp(app->tools[i].uid, "LEGACY") == 0) {
                generate_uid(app->tools[i].uid, sizeof(app->tools[i].uid));

                // Set created timestamp if not already set
                if (app->tools[i].created == 0) {
                    app->tools[i].created = time(NULL);
                }

                generated++;
            }
        }

        set_modified(app, 1);
        update_display(app);

        char status[256];
        snprintf(status, sizeof(status),
                 "Generated unique IDs for %d tool%s - click Save to keep changes",
                 generated, generated == 1 ? "" : "s");
        gtk_label_set_text(GTK_LABEL(app->status_label), status);
    }
}

// Export retired tools to CSV
static void on_export_history_csv(GtkWidget *widget, gpointer data) {
    (void)widget;
    GtkWidget *retired_list = (GtkWidget *)data;
    GtkTreeModel *model = gtk_tree_view_get_model(GTK_TREE_VIEW(retired_list));

    // File chooser dialog - compact size so buttons are reachable
    GtkWidget *dialog = gtk_file_chooser_dialog_new("Export Retired Tools to CSV",
        NULL, GTK_FILE_CHOOSER_ACTION_SAVE,
        "Cancel", GTK_RESPONSE_CANCEL,
        "Save", GTK_RESPONSE_ACCEPT,
        NULL);

    // Set compact size so dialog fits on screen and buttons are accessible
    gtk_window_set_default_size(GTK_WINDOW(dialog), 600, 400);

    gtk_file_chooser_set_do_overwrite_confirmation(GTK_FILE_CHOOSER(dialog), TRUE);
    gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(dialog), "retired_tools.csv");

    if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT) {
        char *filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));

        FILE *f = fopen(filename, "w");
        if (f) {
            // Write CSV header
            fprintf(f, "Tool No,Pocket,Type,UID,Status,Created,Retired,Retire Reason,Retire Notes,");
            fprintf(f, "X,Y,Z,A,B,C,U,V,W,Diameter,Front Angle,Back Angle,Orientation,Usage (sec),Comment\n");

            // Write each row
            GtkTreeIter iter;
            gboolean valid = gtk_tree_model_get_iter_first(model, &iter);
            while (valid) {
                int tool_no, pocket, orientation;
                double x, y, z, a, b, c, u, v, w, diameter, frontangle, backangle, usage;
                char *type, *uid, *status, *created_str, *retired_str, *reason, *notes, *comment;

                gtk_tree_model_get(model, &iter,
                    0, &tool_no, 1, &pocket, 2, &type, 3, &uid, 4, &status,
                    5, &created_str, 6, &retired_str, 7, &reason, 8, &notes,
                    9, &x, 10, &y, 11, &z, 12, &a, 13, &b, 14, &c,
                    15, &u, 16, &v, 17, &w, 18, &diameter,
                    19, &frontangle, 20, &backangle, 21, &orientation, 22, &usage,
                    23, &comment, -1);

                fprintf(f, "%d,%d,\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",",
                        tool_no, pocket, type, uid, status, created_str, retired_str, reason, notes);
                fprintf(f, "%.4f,%.4f,%.4f,%.4f,%.4f,%.4f,%.4f,%.4f,%.4f,%.4f,%.4f,%.4f,%d,%.1f,\"%s\"\n",
                        x, y, z, a, b, c, u, v, w, diameter, frontangle, backangle, orientation, usage, comment);

                g_free(type); g_free(uid); g_free(status); g_free(created_str);
                g_free(retired_str); g_free(reason); g_free(notes); g_free(comment);

                valid = gtk_tree_model_iter_next(model, &iter);
            }

            fclose(f);

            GtkWidget *msg = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL,
                GTK_MESSAGE_INFO, GTK_BUTTONS_OK,
                "Exported retired tools to:\n%s", filename);
            gtk_dialog_run(GTK_DIALOG(msg));
            gtk_widget_destroy(msg);
        } else {
            GtkWidget *err = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL,
                GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
                "Error: Could not write to file");
            gtk_dialog_run(GTK_DIALOG(err));
            gtk_widget_destroy(err);
        }

        g_free(filename);
    }

    gtk_widget_destroy(dialog);
}

static void on_history_clicked(GtkWidget *widget, gpointer data) {
    (void)widget;
    ManagerApp *app = (ManagerApp *)data;

    // Build history file path
    char history_file[600];
    snprintf(history_file, sizeof(history_file), "%s.history", app->db_file);

    // Create new window for retired tools
    GtkWidget *history_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(history_window), "Retired Tools History");
    gtk_window_set_default_size(GTK_WINDOW(history_window), 1200, 400);
    gtk_window_set_transient_for(GTK_WINDOW(history_window), GTK_WINDOW(app->window));

    // Main vertical box - compact spacing to match main tool table
    GtkWidget *vbox = gtk_box_new(GTK_ORIENTATION_VERTICAL, 4);
    gtk_container_add(GTK_CONTAINER(history_window), vbox);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);

    // Toolbar with export button - compact spacing
    GtkWidget *toolbar = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 5);
    gtk_box_pack_start(GTK_BOX(vbox), toolbar, FALSE, FALSE, 0);

    GtkWidget *label = gtk_label_new(NULL);
    char label_text[700];
    snprintf(label_text, sizeof(label_text), "<b>Retired Tools from:</b> %s", history_file);
    gtk_label_set_markup(GTK_LABEL(label), label_text);
    gtk_box_pack_start(GTK_BOX(toolbar), label, FALSE, FALSE, 0);

    // Create list store for retired tools (24 columns)
    GtkListStore *retired_store = gtk_list_store_new(24,
        G_TYPE_INT, G_TYPE_INT,  // Tool No, Pocket
        G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING,  // Type, UID, Status
        G_TYPE_STRING, G_TYPE_STRING,  // Created, Retired (as strings)
        G_TYPE_STRING, G_TYPE_STRING,  // Retire Reason, Retire Notes
        G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_DOUBLE,  // X, Y, Z
        G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_DOUBLE,  // A, B, C
        G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_DOUBLE,  // U, V, W
        G_TYPE_DOUBLE,  // Diameter
        G_TYPE_DOUBLE, G_TYPE_DOUBLE,  // Front angle, Back angle
        G_TYPE_INT,  // Orientation
        G_TYPE_DOUBLE,  // Usage
        G_TYPE_STRING);  // Comment

    // Load retired tools from history file
    FILE *f = fopen(history_file, "r");
    int retired_count = 0;
    if (f) {
        char line[1024];
        while (fgets(line, sizeof(line), f)) {
            if (line[0] == ';' || line[0] == '\n' || line[0] == '#') continue;

            Tool t;
            memset(&t, 0, sizeof(Tool));
            strcpy(t.status, "RETIRED");
            strcpy(t.type, "UNKNOWN");
            strcpy(t.uid, "LEGACY");

            // Parse the tool line (reuse existing parser logic)
            char temp[1024];
            strncpy(temp, line, sizeof(temp) - 1);
            char *comment_start = strchr(temp, ';');
            if (comment_start) {
                comment_start++;
                while (*comment_start == ' ') comment_start++;
                strncpy(t.comment, comment_start, sizeof(t.comment) - 1);
                char *nl = strchr(t.comment, '\n');
                if (nl) *nl = '\0';
            }
            if (comment_start) *comment_start = '\0';

            char *token = strtok(temp, " \t\n");
            while (token != NULL) {
                if (strncmp(token, "UID:", 4) == 0) {
                    strncpy(t.uid, token + 4, sizeof(t.uid) - 1);
                } else if (strncmp(token, "TYPE:", 5) == 0) {
                    strncpy(t.type, token + 5, sizeof(t.type) - 1);
                } else if (strncmp(token, "STATUS:", 7) == 0) {
                    strncpy(t.status, token + 7, sizeof(t.status) - 1);
                } else if (strncmp(token, "CREATED:", 8) == 0) {
                    t.created = (time_t)atol(token + 8);
                } else if (strncmp(token, "RETIRED:", 8) == 0) {
                    t.retired = (time_t)atol(token + 8);
                } else if (strncmp(token, "RETIRE_REASON:", 14) == 0) {
                    strncpy(t.retire_reason, token + 14, sizeof(t.retire_reason) - 1);
                    // Convert underscores back to spaces
                    for (char *p = t.retire_reason; *p; p++) {
                        if (*p == '_') *p = ' ';
                    }
                } else if (strncmp(token, "RETIRE_NOTES:", 13) == 0) {
                    strncpy(t.retire_notes, token + 13, sizeof(t.retire_notes) - 1);
                    // Convert underscores back to spaces
                    for (char *p = t.retire_notes; *p; p++) {
                        if (*p == '_') *p = ' ';
                    }
                } else if (strlen(token) > 1) {
                    char letter = token[0];
                    if (letter == 'T') t.tool_no = atoi(token + 1);
                    else if (letter == 'P') t.pocket = atoi(token + 1);
                    else if (letter == 'X') t.x = atof(token + 1);
                    else if (letter == 'Y') t.y = atof(token + 1);
                    else if (letter == 'Z') t.z = atof(token + 1);
                    else if (letter == 'A') t.a = atof(token + 1);
                    else if (letter == 'B') t.b = atof(token + 1);
                    else if (letter == 'C') t.c = atof(token + 1);
                    else if (letter == 'U') t.u = atof(token + 1);
                    else if (letter == 'V') t.v = atof(token + 1);
                    else if (letter == 'W') t.w = atof(token + 1);
                    else if (letter == 'D') t.diameter = atof(token + 1);
                    else if (letter == 'I') t.frontangle = atof(token + 1);
                    else if (letter == 'J') t.backangle = atof(token + 1);
                    else if (letter == 'Q') t.orientation = atoi(token + 1);
                    else if (letter == 'M') t.usage_seconds = atof(token + 1);
                }
                token = strtok(NULL, " \t\n");
            }

            // Add to list store
            GtkTreeIter iter;
            gtk_list_store_append(retired_store, &iter);

            char created_str[64], retired_str[64];
            if (t.created > 0) {
                struct tm tm_buf;
                strftime(created_str, sizeof(created_str), "%Y-%m-%d %H:%M", localtime_r(&t.created, &tm_buf));
            } else {
                strcpy(created_str, "N/A");
            }
            if (t.retired > 0) {
                struct tm tm_buf;
                strftime(retired_str, sizeof(retired_str), "%Y-%m-%d %H:%M", localtime_r(&t.retired, &tm_buf));
            } else {
                strcpy(retired_str, "N/A");
            }

            gtk_list_store_set(retired_store, &iter,
                0, t.tool_no, 1, t.pocket, 2, t.type, 3, t.uid, 4, t.status,
                5, created_str, 6, retired_str, 7, t.retire_reason, 8, t.retire_notes,
                9, t.x, 10, t.y, 11, t.z, 12, t.a, 13, t.b, 14, t.c,
                15, t.u, 16, t.v, 17, t.w, 18, t.diameter,
                19, t.frontangle, 20, t.backangle, 21, t.orientation, 22, t.usage_seconds,
                23, t.comment, -1);

            retired_count++;
        }
        fclose(f);
    }

    // Scrolled window for table - compact like main table
    GtkWidget *scrolled = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_set_overlay_scrolling(GTK_SCROLLED_WINDOW(scrolled), FALSE);
    gtk_container_set_border_width(GTK_CONTAINER(scrolled), 2);
    gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);

    // Create tree view with compact spacing
    GtkWidget *tree_view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(retired_store));
    g_object_unref(retired_store);
    gtk_tree_view_set_grid_lines(GTK_TREE_VIEW(tree_view), GTK_TREE_VIEW_GRID_LINES_BOTH);

    // Make rows more compact - match main tool table styling
    GtkCssProvider *css_provider = gtk_css_provider_new();
    gtk_css_provider_load_from_data(css_provider,
        "treeview { -GtkTreeView-vertical-separator: 0; margin-bottom: 4px; }"
        "treeview row { padding: 1px; min-height: 20px; }",
        -1, NULL);
    gtk_style_context_add_provider(gtk_widget_get_style_context(tree_view),
        GTK_STYLE_PROVIDER(css_provider), GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
    g_object_unref(css_provider);

    gtk_container_add(GTK_CONTAINER(scrolled), tree_view);

    // Add columns with compact styling to match main table
    const char *col_titles[] = {"Tool", "Pocket", "Type", "UID", "Status", "Created", "Retired",
                                 "Retire Reason", "Retire Notes", "X", "Y", "Z", "A", "B", "C",
                                 "U", "V", "W", "Diameter", "FrontAngle", "BackAngle", "Orient", "Usage", "Comment"};
    for (int i = 0; i < 24; i++) {
        GtkCellRenderer *renderer = gtk_cell_renderer_text_new();
        // Reduce cell padding for compactness
        g_object_set(renderer, "ypad", 0, "xpad", 2, NULL);
        GtkTreeViewColumn *column = gtk_tree_view_column_new_with_attributes(
            col_titles[i], renderer, "text", i, NULL);
        gtk_tree_view_column_set_resizable(column, TRUE);
        gtk_tree_view_column_set_sort_column_id(column, i);
        gtk_tree_view_column_set_min_width(column, 50);
        gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), column);
    }

    // Bottom toolbar with count and export button - compact spacing
    GtkWidget *bottom_toolbar = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 5);
    gtk_box_pack_start(GTK_BOX(vbox), bottom_toolbar, FALSE, FALSE, 2);

    char count_text[200];
    snprintf(count_text, sizeof(count_text), "Total retired tools: %d", retired_count);
    GtkWidget *count_label = gtk_label_new(count_text);
    gtk_box_pack_start(GTK_BOX(bottom_toolbar), count_label, FALSE, FALSE, 0);

    GtkWidget *spacer = gtk_label_new("");
    gtk_box_pack_start(GTK_BOX(bottom_toolbar), spacer, TRUE, TRUE, 0);

    GtkWidget *export_button = gtk_button_new_with_label("Export to CSV");
    g_signal_connect(export_button, "clicked", G_CALLBACK(on_export_history_csv), tree_view);
    gtk_box_pack_start(GTK_BOX(bottom_toolbar), export_button, FALSE, FALSE, 0);

    GtkWidget *close_button = gtk_button_new_with_label("Close");
    g_signal_connect_swapped(close_button, "clicked", G_CALLBACK(gtk_widget_destroy), history_window);
    gtk_box_pack_start(GTK_BOX(bottom_toolbar), close_button, FALSE, FALSE, 0);

    gtk_widget_show_all(history_window);

    if (retired_count == 0) {
        GtkWidget *msg = gtk_message_dialog_new(GTK_WINDOW(history_window),
            GTK_DIALOG_MODAL, GTK_MESSAGE_INFO, GTK_BUTTONS_OK,
            "No retired tools found in history file.");
        gtk_dialog_run(GTK_DIALOG(msg));
        gtk_widget_destroy(msg);
    }
}

static void on_stats_clicked(GtkWidget *widget, gpointer data) {
    (void)widget;
    ManagerApp *app = (ManagerApp *)data;

    // Calculate type statistics
    typedef struct {
        char type[64];
        int count;
        double total_usage;
        double avg_usage;
    } TypeStat;

    TypeStat stats[50];
    int stat_count = 0;

    // Count tools by type
    for (int i = 0; i < app->tool_count; i++) {
        Tool *t = &app->tools[i];
        int found = -1;

        for (int j = 0; j < stat_count; j++) {
            if (strcmp(stats[j].type, t->type) == 0) {
                found = j;
                break;
            }
        }

        if (found >= 0) {
            stats[found].count++;
            stats[found].total_usage += t->usage_seconds;
        } else if (stat_count < 50) {
            strcpy(stats[stat_count].type, t->type);
            stats[stat_count].count = 1;
            stats[stat_count].total_usage = t->usage_seconds;
            stat_count++;
        }
    }

    // Calculate averages
    for (int i = 0; i < stat_count; i++) {
        stats[i].avg_usage = stats[i].total_usage / stats[i].count;
    }

    // Build message
    char msg[2048];
    snprintf(msg, sizeof(msg), "Tool Type Statistics\n\n");

    for (int i = 0; i < stat_count; i++) {
        char line[256];
        snprintf(line, sizeof(line), "%s: %d tools, Avg: %.1f sec, Total: %.1f sec\n",
                 stats[i].type, stats[i].count, stats[i].avg_usage, stats[i].total_usage);
        strncat(msg, line, sizeof(msg) - strlen(msg) - 1);
    }

    if (stat_count == 0) {
        snprintf(msg, sizeof(msg), "No tool statistics available yet.");
    }

    GtkWidget *dialog = gtk_message_dialog_new(GTK_WINDOW(app->window),
        GTK_DIALOG_MODAL, GTK_MESSAGE_INFO, GTK_BUTTONS_OK,
        "%s", msg);
    gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_destroy(dialog);
}

static void on_refresh_clicked(GtkWidget *widget, gpointer data) {
    (void)widget;
    update_data(data);
    gtk_label_set_text(GTK_LABEL(((ManagerApp*)data)->status_label), "Refreshed from disk");
}

// Periodic update
static gboolean update_data(gpointer data) {
    ManagerApp *app = (ManagerApp *)data;
    struct stat st;

    // Check if file was modified externally
    // Check both mtime AND size to catch changes within same second (mtime granularity)
    if (stat(app->db_file, &st) == 0) {
        if (st.st_mtime != app->last_mtime || st.st_size != app->last_size) {
            app->last_mtime = st.st_mtime;
            app->last_size = st.st_size;

            if (!app->modified) {
                // Reload if we haven't made local changes
                parse_database(app);
                for (int i = 0; i < app->tool_count; i++) {
                    if (app->tools[i].tool_no == app->current_tool) {
                        app->tools[i].load_time = app->current_tool_load_time;
                    } else {
                        app->tools[i].load_time = 0;
                    }
                }
            }
        }
    }

    update_display(app);
    return TRUE;
}

// Add editable column helper
static void add_column(ManagerApp *app, const char *title, int col_id, gboolean editable, gboolean is_int) {
    GtkCellRenderer *renderer = gtk_cell_renderer_text_new();

    if (editable) {
        g_object_set(renderer, "editable", TRUE, NULL);
        g_object_set_data(G_OBJECT(renderer), "column", GINT_TO_POINTER(col_id));
        g_signal_connect(renderer, "edited", G_CALLBACK(on_cell_edited), app);
        g_signal_connect(renderer, "editing-started", G_CALLBACK(on_editing_started), app);
        g_signal_connect(renderer, "editing-canceled", G_CALLBACK(on_editing_canceled), app);
    }

    if (!is_int) {
        g_object_set(renderer, "xalign", 1.0, NULL);
    }

    GtkTreeViewColumn *column = gtk_tree_view_column_new_with_attributes(title, renderer, NULL);
    gtk_tree_view_column_set_resizable(column, TRUE);
    gtk_tree_view_column_set_sort_column_id(column, col_id);
    gtk_tree_view_column_set_min_width(column, 60);

    // Set cell data function for formatting
    gtk_tree_view_column_set_cell_data_func(column, renderer, number_cell_data_func,
                                           GINT_TO_POINTER(col_id), NULL);

    gtk_tree_view_append_column(GTK_TREE_VIEW(app->tree_view), column);
}

// Create UI
static void create_ui(ManagerApp *app) {
    GtkWidget *vbox, *toolbar, *scrolled, *button, *sep, *info_box;
    char title[600];

    // Main vertical box - compact spacing
    vbox = gtk_box_new(GTK_ORIENTATION_VERTICAL, 4);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);
    gtk_container_add(GTK_CONTAINER(app->window), vbox);

    // Info box at top
    info_box = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 10);
    gtk_box_pack_start(GTK_BOX(vbox), info_box, FALSE, FALSE, 0);

    app->current_tool_label = gtk_label_new(NULL);
    gtk_label_set_markup(GTK_LABEL(app->current_tool_label), "<b>Current Tool:</b> None");
    gtk_box_pack_start(GTK_BOX(info_box), app->current_tool_label, FALSE, FALSE, 0);

    app->running_label = gtk_label_new(NULL);
    gtk_label_set_markup(GTK_LABEL(app->running_label), "<b>RUNNING:</b> <span foreground='red'>OFF</span>");
    gtk_box_pack_start(GTK_BOX(info_box), app->running_label, FALSE, FALSE, 15);

    gtk_box_pack_start(GTK_BOX(info_box), gtk_label_new(""), TRUE, TRUE, 0);

    app->update_label = gtk_label_new(NULL);
    gtk_label_set_markup(GTK_LABEL(app->update_label), "<small>Updated: Never</small>");
    gtk_box_pack_start(GTK_BOX(info_box), app->update_label, FALSE, FALSE, 0);

    // Toolbar
    toolbar = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 5);
    gtk_box_pack_start(GTK_BOX(vbox), toolbar, FALSE, FALSE, 0);

    button = gtk_button_new_with_label("Add Tool");
    g_signal_connect(button, "clicked", G_CALLBACK(on_add_tool_clicked), app);
    gtk_box_pack_start(GTK_BOX(toolbar), button, FALSE, FALSE, 0);

    button = gtk_button_new_with_label("Delete Tool");
    g_signal_connect(button, "clicked", G_CALLBACK(on_delete_tool_clicked), app);
    gtk_box_pack_start(GTK_BOX(toolbar), button, FALSE, FALSE, 0);

    button = gtk_button_new_with_label("Retire Tool");
    g_signal_connect(button, "clicked", G_CALLBACK(on_retire_tool_clicked), app);
    gtk_box_pack_start(GTK_BOX(toolbar), button, FALSE, FALSE, 0);

    sep = gtk_separator_new(GTK_ORIENTATION_VERTICAL);
    gtk_box_pack_start(GTK_BOX(toolbar), sep, FALSE, FALSE, 5);

    button = gtk_button_new_with_label("Generate UIDs");
    g_signal_connect(button, "clicked", G_CALLBACK(on_generate_uids_clicked), app);
    gtk_box_pack_start(GTK_BOX(toolbar), button, FALSE, FALSE, 0);

    sep = gtk_separator_new(GTK_ORIENTATION_VERTICAL);
    gtk_box_pack_start(GTK_BOX(toolbar), sep, FALSE, FALSE, 5);

    app->save_button = gtk_button_new_with_label("Save");
    g_signal_connect(app->save_button, "clicked", G_CALLBACK(on_save_clicked), app);
    gtk_box_pack_start(GTK_BOX(toolbar), app->save_button, FALSE, FALSE, 0);
    gtk_widget_set_sensitive(app->save_button, FALSE);

    app->reload_button = gtk_button_new_with_label("Reload to LinuxCNC");
    g_signal_connect(app->reload_button, "clicked", G_CALLBACK(on_reload_clicked), app);
    gtk_box_pack_start(GTK_BOX(toolbar), app->reload_button, FALSE, FALSE, 0);

    sep = gtk_separator_new(GTK_ORIENTATION_VERTICAL);
    gtk_box_pack_start(GTK_BOX(toolbar), sep, FALSE, FALSE, 5);

    button = gtk_button_new_with_label("Refresh");
    g_signal_connect(button, "clicked", G_CALLBACK(on_refresh_clicked), app);
    gtk_box_pack_start(GTK_BOX(toolbar), button, FALSE, FALSE, 0);

    sep = gtk_separator_new(GTK_ORIENTATION_VERTICAL);
    gtk_box_pack_start(GTK_BOX(toolbar), sep, FALSE, FALSE, 5);

    button = gtk_button_new_with_label("History");
    g_signal_connect(button, "clicked", G_CALLBACK(on_history_clicked), app);
    gtk_box_pack_start(GTK_BOX(toolbar), button, FALSE, FALSE, 0);

    button = gtk_button_new_with_label("Stats");
    g_signal_connect(button, "clicked", G_CALLBACK(on_stats_clicked), app);
    gtk_box_pack_start(GTK_BOX(toolbar), button, FALSE, FALSE, 0);

    // Scrolled window - limit height to show ~10 rows
    scrolled = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    // Disable overlay scrollbars to prevent overlap with content
    gtk_scrolled_window_set_overlay_scrolling(GTK_SCROLLED_WINDOW(scrolled), FALSE);
    // Add bottom padding to prevent scrollbar overlap
    gtk_container_set_border_width(GTK_CONTAINER(scrolled), 2);
    // Set size for ~10 compact rows (height: 250px, min width: 1100px for all columns)
    gtk_widget_set_size_request(scrolled, 1100, 250);
    // Don't expand vertically - prevents extra white space
    gtk_box_pack_start(GTK_BOX(vbox), scrolled, FALSE, FALSE, 0);

    // Create list store
    app->store = gtk_list_store_new(NUM_COLS,
        G_TYPE_INT, G_TYPE_INT,  // Tool, Pocket
        G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_DOUBLE,  // X, Y, Z
        G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_DOUBLE,  // A, B, C
        G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_DOUBLE,  // U, V, W
        G_TYPE_DOUBLE,  // Diameter
        G_TYPE_DOUBLE, G_TYPE_DOUBLE,  // Front angle, Back angle
        G_TYPE_INT,  // Orientation
        G_TYPE_DOUBLE,  // Usage
        G_TYPE_STRING,  // Type
        G_TYPE_STRING,  // UID
        G_TYPE_STRING,  // Status
        G_TYPE_STRING,  // Comment
        G_TYPE_BOOLEAN  // Is current
    );

    // Tree view with compact spacing
    app->tree_view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(app->store));
    gtk_tree_view_set_grid_lines(GTK_TREE_VIEW(app->tree_view), GTK_TREE_VIEW_GRID_LINES_BOTH);

    // Make rows more compact - reduce vertical padding and add bottom margin for scrollbar
    GtkCssProvider *css_provider = gtk_css_provider_new();
    gtk_css_provider_load_from_data(css_provider,
        "treeview { -GtkTreeView-vertical-separator: 0; margin-bottom: 4px; }"
        "treeview row { padding: 1px; min-height: 20px; }",
        -1, NULL);
    gtk_style_context_add_provider(gtk_widget_get_style_context(app->tree_view),
        GTK_STYLE_PROVIDER(css_provider), GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
    g_object_unref(css_provider);

    gtk_container_add(GTK_CONTAINER(scrolled), app->tree_view);

    // Add all columns (editable except Usage)
    add_column(app, "Tool", COL_TOOL_NO, TRUE, TRUE);
    add_column(app, "Pocket", COL_POCKET, TRUE, TRUE);
    add_column(app, "X", COL_X, TRUE, FALSE);
    add_column(app, "Y", COL_Y, TRUE, FALSE);
    add_column(app, "Z", COL_Z, TRUE, FALSE);
    add_column(app, "A", COL_A, TRUE, FALSE);
    add_column(app, "B", COL_B, TRUE, FALSE);
    add_column(app, "C", COL_C, TRUE, FALSE);
    add_column(app, "U", COL_U, TRUE, FALSE);
    add_column(app, "V", COL_V, TRUE, FALSE);
    add_column(app, "W", COL_W, TRUE, FALSE);
    add_column(app, "Diam", COL_DIAMETER, TRUE, FALSE);
    add_column(app, "∠I", COL_FRONTANGLE, TRUE, FALSE);
    add_column(app, "∠J", COL_BACKANGLE, TRUE, FALSE);
    add_column(app, "Q", COL_ORIENTATION, TRUE, TRUE);
    add_column(app, "Usage(sec)", COL_USAGE, FALSE, FALSE);

    // Type column (dropdown with predefined types + custom entry)
    app->type_store = gtk_list_store_new(1, G_TYPE_STRING);
    GtkTreeIter type_iter;

    // Populate dropdown with predefined types
    for (int i = 0; TOOL_TYPES[i] != NULL; i++) {
        gtk_list_store_append(app->type_store, &type_iter);
        gtk_list_store_set(app->type_store, &type_iter, 0, TOOL_TYPES[i], -1);
    }

    GtkCellRenderer *type_renderer = gtk_cell_renderer_combo_new();
    g_object_set(type_renderer,
                 "editable", TRUE,
                 "model", app->type_store,
                 "text-column", 0,
                 "has-entry", TRUE,  // Allow custom entries
                 NULL);
    // Note: Don't unref type_store here, we need it to add custom types later

    g_object_set_data(G_OBJECT(type_renderer), "column", GINT_TO_POINTER(COL_TYPE));
    g_signal_connect(type_renderer, "edited", G_CALLBACK(on_cell_edited), app);
    g_signal_connect(type_renderer, "editing-started", G_CALLBACK(on_editing_started), app);
    g_signal_connect(type_renderer, "editing-canceled", G_CALLBACK(on_editing_canceled), app);

    GtkTreeViewColumn *type_column = gtk_tree_view_column_new_with_attributes(
        "Type", type_renderer, "text", COL_TYPE, NULL);
    gtk_tree_view_column_set_resizable(type_column, TRUE);
    gtk_tree_view_column_set_min_width(type_column, 120);
    gtk_tree_view_column_set_cell_data_func(type_column, type_renderer, text_cell_data_func, NULL, NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(app->tree_view), type_column);

    // UID column (read-only text)
    GtkCellRenderer *uid_renderer = gtk_cell_renderer_text_new();
    g_object_set(uid_renderer, "editable", FALSE, NULL);
    g_object_set(uid_renderer, "foreground", "#666666", NULL);  // Gray text

    GtkTreeViewColumn *uid_column = gtk_tree_view_column_new_with_attributes(
        "UID", uid_renderer, "text", COL_UID, NULL);
    gtk_tree_view_column_set_resizable(uid_column, TRUE);
    gtk_tree_view_column_set_min_width(uid_column, 140);
    gtk_tree_view_column_set_cell_data_func(uid_column, uid_renderer, text_cell_data_func, NULL, NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(app->tree_view), uid_column);

    // Status column (read-only text) - color-coded
    GtkCellRenderer *status_renderer = gtk_cell_renderer_text_new();
    g_object_set(status_renderer, "editable", FALSE, NULL);

    GtkTreeViewColumn *status_column = gtk_tree_view_column_new_with_attributes(
        "Status", status_renderer, "text", COL_STATUS, NULL);
    gtk_tree_view_column_set_resizable(status_column, TRUE);
    gtk_tree_view_column_set_min_width(status_column, 80);
    gtk_tree_view_column_set_cell_data_func(status_column, status_renderer, text_cell_data_func, NULL, NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(app->tree_view), status_column);

    // Comment column (special handling for text)
    GtkCellRenderer *renderer = gtk_cell_renderer_text_new();
    g_object_set(renderer, "editable", TRUE, NULL);
    g_object_set_data(G_OBJECT(renderer), "column", GINT_TO_POINTER(COL_COMMENT));
    g_signal_connect(renderer, "edited", G_CALLBACK(on_cell_edited), app);
    g_signal_connect(renderer, "editing-started", G_CALLBACK(on_editing_started), app);
    g_signal_connect(renderer, "editing-canceled", G_CALLBACK(on_editing_canceled), app);

    GtkTreeViewColumn *column = gtk_tree_view_column_new_with_attributes(
        "Comment", renderer, "text", COL_COMMENT, NULL);
    gtk_tree_view_column_set_resizable(column, TRUE);
    gtk_tree_view_column_set_min_width(column, 200);
    gtk_tree_view_column_set_cell_data_func(column, renderer, text_cell_data_func, NULL, NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(app->tree_view), column);

    // Statistics bar
    app->stats_label = gtk_label_new(NULL);
    gtk_label_set_markup(GTK_LABEL(app->stats_label), "<i>Loading...</i>");
    gtk_label_set_xalign(GTK_LABEL(app->stats_label), 0);
    gtk_box_pack_start(GTK_BOX(vbox), app->stats_label, FALSE, FALSE, 0);

    // Status bar
    app->status_label = gtk_label_new(NULL);
    gtk_label_set_markup(GTK_LABEL(app->status_label), "<i>Ready - Edit any cell to modify values</i>");
    gtk_label_set_xalign(GTK_LABEL(app->status_label), 0);
    gtk_box_pack_start(GTK_BOX(vbox), app->status_label, FALSE, FALSE, 0);

    snprintf(title, sizeof(title), "LinuxCNC Tool Manager - %s",
             strrchr(app->db_file, '/') ? strrchr(app->db_file, '/') + 1 : app->db_file);
    gtk_window_set_title(GTK_WINDOW(app->window), title);
}

// Find active database
static int find_active_database(char *db_path, size_t path_size) {
    FILE *fp;
    char line[1024];
    char proc_path[256];
    char cmdline[1024];
    int found = 0;

    fp = popen("pgrep -x db_tool", "r");
    if (fp) {
        while (fgets(line, sizeof(line), fp)) {
            int pid = atoi(line);
            if (pid > 0) {
                snprintf(proc_path, sizeof(proc_path), "/proc/%d/cmdline", pid);
                FILE *cmdline_file = fopen(proc_path, "r");
                if (cmdline_file) {
                    size_t len = fread(cmdline, 1, sizeof(cmdline) - 1, cmdline_file);
                    fclose(cmdline_file);

                    if (len > 0) {
                        cmdline[len] = '\0';
                        for (size_t i = 0; i < len; i++) {
                            if (cmdline[i] == '\0') cmdline[i] = ' ';
                        }

                        char *dat_pos = strstr(cmdline, ".dat");
                        if (dat_pos) {
                            char *start = dat_pos;
                            while (start > cmdline && *(start - 1) != ' ') start--;

                            size_t path_len = dat_pos + 4 - start;
                            if (path_len > 0 && path_len < path_size) {
                                strncpy(db_path, start, path_len);
                                db_path[path_len] = '\0';

                                if (access(db_path, R_OK) == 0) {
                                    found = 1;
                                    break;
                                }
                            }
                        }
                    }
                }
            }
        }
        pclose(fp);
    }

    if (found) return 1;

    // Fallback
    char *home = getenv("HOME");
    if (home) {
        char search_dir[512];
        snprintf(search_dir, sizeof(search_dir), "%s/linuxcnc_tool_data", home);
        if (access(search_dir, R_OK) == 0) {
            char cmd[1024];
            snprintf(cmd, sizeof(cmd),
                    "find %s -maxdepth 1 -name '*.dat' -type f -printf '%%T@ %%p\\n' 2>/dev/null | sort -rn | head -1 | cut -d' ' -f2-",
                    search_dir);

            fp = popen(cmd, "r");
            if (fp) {
                if (fgets(line, sizeof(line), fp)) {
                    line[strcspn(line, "\n")] = '\0';
                    if (strlen(line) > 0 && access(line, R_OK) == 0) {
                        strncpy(db_path, line, path_size - 1);
                        db_path[path_size - 1] = '\0';
                        found = 1;
                    }
                }
                pclose(fp);
            }
        }
    }

    return found;
}

int main(int argc, char *argv[]) {
    ManagerApp app;
    char *home;

    memset(&app, 0, sizeof(app));
    app.current_tool = -1;
    app.prev_current_tool = -1;
    app.modified = 0;
    app.auto_reload = 1;

    // Get database file
    if (argc > 1) {
        strncpy(app.db_file, argv[1], sizeof(app.db_file) - 1);
    } else {
        fprintf(stderr, "Auto-detecting database...\n");
        if (find_active_database(app.db_file, sizeof(app.db_file))) {
            fprintf(stderr, "Found: %s\n", app.db_file);
        } else {
            home = getenv("HOME");
            if (home) {
                snprintf(app.db_file, sizeof(app.db_file),
                        "%s/linuxcnc_tool_data/tool_usage.dat", home);
            } else {
                fprintf(stderr, "Error: Cannot determine home directory\n");
                return 1;
            }
        }
    }

    if (access(app.db_file, R_OK) != 0) {
        fprintf(stderr, "Error: Database file not found: %s\n", app.db_file);
        return 1;
    }

    gtk_init(&argc, &argv);

    app.window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    // Compact window - size to fit content without extra white space
    gtk_window_set_default_size(GTK_WINDOW(app.window), 1200, 400);
    gtk_window_set_resizable(GTK_WINDOW(app.window), TRUE);
    g_signal_connect(app.window, "destroy", G_CALLBACK(gtk_main_quit), NULL);

    create_ui(&app);
    parse_database(&app);

    // Add all existing tool types to dropdown
    for (int i = 0; i < app.tool_count; i++) {
        add_type_to_dropdown(&app, app.tools[i].type);
    }

    // Trust the database header for current tool - db_tool writes it immediately on every tool change
    // The database is the single source of truth for tool state
    fprintf(stderr, "Using database current tool: T%d\n", app.current_tool);

    // Set load_time for current tool
    for (int i = 0; i < app.tool_count; i++) {
        if (app.tools[i].tool_no == app.current_tool) {
            app.tools[i].load_time = app.current_tool_load_time;
        } else {
            app.tools[i].load_time = 0;
        }
    }

    update_display(&app);

    // Auto-update every second for real-time monitoring
    g_timeout_add(1000, update_data, &app);

    gtk_widget_show_all(app.window);
    gtk_main();

    return 0;
}
