/*
 * LinuxCNC Tool Usage Monitor - GTK3 GUI in C
 * Real-time graphical display of tool usage times
 *
 * Compile: gcc -o db_tool_monitor db_tool_monitor.c `pkg-config --cflags --libs gtk+-3.0` -lm
 * Usage: db_tool_monitor [database_file]
 */

#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/stat.h>
#include <unistd.h>
#include <math.h>

#define MAX_TOOLS 200
#define LINE_LEN 512

typedef struct {
    int tool_no;
    int pocket;
    double z_offset;
    double diameter;
    double usage_minutes;
    char comment[128];
    int is_current;
    time_t load_time;  // When tool was loaded (for live calculation)
} Tool;

typedef struct {
    GtkWidget *window;
    GtkWidget *tree_view;
    GtkListStore *store;
    GtkWidget *current_tool_label;
    GtkWidget *update_label;
    GtkWidget *status_label;

    char db_file[512];
    int current_tool;
    int prev_current_tool;  // Previous current tool (detect changes)
    time_t current_tool_load_time;  // When current tool was loaded
    time_t last_mtime;

    Tool tools[MAX_TOOLS];
    int tool_count;
    int updating;  // Flag to prevent selection changes during update
} MonitorApp;

// Function prototypes
static gboolean update_data(gpointer data);
static void on_refresh_clicked(GtkWidget *widget, gpointer data);
static int parse_database(MonitorApp *app);
static int detect_current_tool(time_t *load_time);
static void update_display(MonitorApp *app);
static void create_ui(MonitorApp *app);

// Parse the database file
static int parse_database(MonitorApp *app) {
    FILE *f;
    char line[LINE_LEN];
    int count = 0;

    f = fopen(app->db_file, "r");
    if (!f) {
        return 0;
    }

    // Reset current tool info
    app->current_tool = 0;
    app->current_tool_load_time = 0;

    while (fgets(line, sizeof(line), f) && count < MAX_TOOLS) {
        // Parse header comments for current tool info
        if (line[0] == ';') {
            // Look for CURRENT_TOOL and LOAD_TIME in header
            if (strstr(line, "; CURRENT_TOOL=") == line) {
                sscanf(line, "; CURRENT_TOOL=%d", &app->current_tool);
            } else if (strstr(line, "; LOAD_TIME=") == line) {
                long load_time_long = 0;
                sscanf(line, "; LOAD_TIME=%ld", &load_time_long);
                app->current_tool_load_time = (time_t)load_time_long;
            }
            continue;  // Skip all comment lines
        }

        // Skip empty lines
        if (line[0] == '\n' || line[0] == '\r') {
            continue;
        }

        Tool *t = &app->tools[count];
        char *comment_start = strchr(line, ';');

        // Parse: T# P# Z#.### D#.### [M#.###] ; comment
        int n = sscanf(line, "T%d P%d Z%lf D%lf M%lf",
                       &t->tool_no, &t->pocket, &t->z_offset,
                       &t->diameter, &t->usage_minutes);

        if (n >= 4) {  // At least T, P, Z, D required
            if (n < 5) {
                t->usage_minutes = 0.0;
            }

            // Extract comment
            if (comment_start) {
                comment_start++;  // Skip ';'
                while (*comment_start == ' ') comment_start++;  // Skip spaces
                strncpy(t->comment, comment_start, sizeof(t->comment) - 1);
                t->comment[sizeof(t->comment) - 1] = '\0';
                // Remove newline
                char *nl = strchr(t->comment, '\n');
                if (nl) *nl = '\0';
                nl = strchr(t->comment, '\r');
                if (nl) *nl = '\0';
            } else {
                snprintf(t->comment, sizeof(t->comment), "Tool_%d", t->tool_no);
            }

            count++;
        }
    }

    fclose(f);
    app->tool_count = count;
    return count;
}

// Detect currently loaded tool from LinuxCNC log and get load time
// Returns: tool number (>0), 0 = no tool loaded, -1 = couldn't detect
static int detect_current_tool(time_t *load_time) {
    FILE *f;
    char line[LINE_LEN];
    char *home = getenv("HOME");
    char log_path[512];
    int current_tool = 0;  // Default: no tool loaded
    struct stat st;

    *load_time = 0;
    if (!home) return -1;

    snprintf(log_path, sizeof(log_path), "%s/.linuxcnc_print.txt", home);

    f = fopen(log_path, "r");
    if (!f) return -1;

    // Get file modification time (approximate time of last log entry)
    if (fstat(fileno(f), &st) == 0) {
        *load_time = st.st_mtime;
    }

    // Read last 100 lines
    char lines[100][LINE_LEN];
    int line_count = 0;

    while (fgets(line, sizeof(line), f)) {
        strncpy(lines[line_count % 100], line, LINE_LEN - 1);
        lines[line_count % 100][LINE_LEN - 1] = '\0';
        line_count++;
    }
    fclose(f);

    // Search backwards for most recent load/unload
    int start = (line_count > 100) ? 100 : line_count;
    for (int i = start - 1; i >= 0; i--) {
        int idx = (line_count > 100) ? ((line_count - 100 + i) % 100) : i;

        if (strstr(lines[idx], ">>> Loaded T")) {
            sscanf(lines[idx], "%*[^T]T%d", &current_tool);
            // Use file mtime as approximate load time
            break;
        } else if (strstr(lines[idx], "<<< Unloaded T")) {
            current_tool = 0;  // No tool loaded
            *load_time = 0;
            break;
        }
    }

    return current_tool;
}

// Update the display
static void update_display(MonitorApp *app) {
    GtkTreeIter iter;
    char markup[256];
    time_t now = time(NULL);
    struct tm *tm_info;
    char time_str[64];
    GtkTreeSelection *selection;
    int selected_tool_no = -1;
    gboolean had_selection = FALSE;
    gint sort_column_id;
    GtkSortType sort_order;
    gboolean was_sorted = FALSE;

    // Set updating flag
    app->updating = 1;

    // Save current selection by tool number (not path, which can change due to sorting)
    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(app->tree_view));
    if (gtk_tree_selection_get_selected(selection, NULL, &iter)) {
        gtk_tree_model_get(GTK_TREE_MODEL(app->store), &iter, 0, &selected_tool_no, -1);
        had_selection = TRUE;
    }

    // Temporarily disable sorting to prevent jumping during value updates
    if (gtk_tree_sortable_get_sort_column_id(GTK_TREE_SORTABLE(app->store),
                                              &sort_column_id, &sort_order)) {
        was_sorted = TRUE;
        gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(app->store),
                                             GTK_TREE_SORTABLE_UNSORTED_SORT_COLUMN_ID,
                                             GTK_SORT_ASCENDING);
    }

    // Update current tool label only if it changed
    if (app->current_tool != app->prev_current_tool) {
        if (app->current_tool > 0) {
            snprintf(markup, sizeof(markup),
                     "<b>Current Tool:</b> <span foreground='green' size='large'>T%d</span>",
                     app->current_tool);
        } else {
            snprintf(markup, sizeof(markup), "<b>Current Tool:</b> None");
        }
        gtk_label_set_markup(GTK_LABEL(app->current_tool_label), markup);
        app->prev_current_tool = app->current_tool;
    }

    // Update rows - only update values that might have changed
    int used_tools = 0;
    double total_usage = 0.0;
    gboolean valid;

    // Get first row
    valid = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(app->store), &iter);

    for (int i = 0; i < app->tool_count; i++) {
        Tool *t = &app->tools[i];
        double display_usage = t->usage_minutes;
        gboolean is_current = (t->tool_no == app->current_tool);

        // If this is the current tool and it has a load time, add live elapsed time
        if (is_current && t->load_time > 0) {
            double elapsed_sec = difftime(now, t->load_time);
            double elapsed_min = elapsed_sec / 60.0;
            display_usage += elapsed_min;
        }

        // Update or insert row
        if (valid) {
            // Only update the columns that change frequently (usage and is_current)
            // This reduces flicker and selection jumping
            gtk_list_store_set(app->store, &iter,
                              4, display_usage,        // Usage changes frequently
                              6, is_current,           // Current tool flag
                              -1);
            // Move to next row
            valid = gtk_tree_model_iter_next(GTK_TREE_MODEL(app->store), &iter);
        } else {
            // No more rows, append new ones (only happens when tool count increases)
            gtk_list_store_append(app->store, &iter);
            gtk_list_store_set(app->store, &iter,
                              0, t->tool_no,
                              1, t->pocket,
                              2, t->z_offset,
                              3, t->diameter,
                              4, display_usage,
                              5, t->comment,
                              6, is_current,
                              -1);
        }

        if (display_usage > 0.001) {
            used_tools++;
            total_usage += display_usage;
        }
    }

    // Remove any extra rows (if tool count decreased)
    while (valid) {
        GtkTreeIter next = iter;
        valid = gtk_tree_model_iter_next(GTK_TREE_MODEL(app->store), &next);
        gtk_list_store_remove(app->store, &iter);
        iter = next;
    }

    // Restore selection if it existed - find the tool by number, not by path
    if (had_selection && selected_tool_no >= 0) {
        gboolean valid = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(app->store), &iter);
        while (valid) {
            int tool_no;
            gtk_tree_model_get(GTK_TREE_MODEL(app->store), &iter, 0, &tool_no, -1);
            if (tool_no == selected_tool_no) {
                gtk_tree_selection_select_iter(selection, &iter);
                break;
            }
            valid = gtk_tree_model_iter_next(GTK_TREE_MODEL(app->store), &iter);
        }
    }

    // Update status label
    snprintf(markup, sizeof(markup),
             "<i>%d tools | %d used | Total usage: %.1f minutes</i>",
             app->tool_count, used_tools, total_usage);
    gtk_label_set_markup(GTK_LABEL(app->status_label), markup);

    // Update timestamp
    now = time(NULL);
    tm_info = localtime(&now);
    strftime(time_str, sizeof(time_str), "%H:%M:%S", tm_info);
    snprintf(markup, sizeof(markup), "<i>Last update: %s</i>", time_str);
    gtk_label_set_markup(GTK_LABEL(app->update_label), markup);

    // Re-enable sorting
    if (was_sorted) {
        gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(app->store),
                                             sort_column_id, sort_order);
    }

    // Clear updating flag
    app->updating = 0;
}

// Cell data function for numeric formatting
static void number_cell_data_func(GtkTreeViewColumn *col,
                                  GtkCellRenderer *renderer,
                                  GtkTreeModel *model,
                                  GtkTreeIter *iter,
                                  gpointer user_data) {
    int col_id = GPOINTER_TO_INT(user_data);
    double value;
    gboolean is_current;
    char text[64];

    gtk_tree_model_get(model, iter, col_id, &value, 6, &is_current, -1);

    if (col_id == 4) {  // Usage column
        if (value > 0.001) {
            snprintf(text, sizeof(text), "%.3f", value);
            g_object_set(renderer, "weight", PANGO_WEIGHT_BOLD, NULL);
        } else {
            snprintf(text, sizeof(text), "0.000");
            g_object_set(renderer, "weight", PANGO_WEIGHT_NORMAL, NULL);
        }
    } else {
        snprintf(text, sizeof(text), "%.4f", value);
    }

    g_object_set(renderer, "text", text, NULL);

    if (is_current) {
        g_object_set(renderer, "cell-background", "#90EE90", NULL);
        g_object_set(renderer, "weight", PANGO_WEIGHT_BOLD, NULL);
    } else {
        g_object_set(renderer, "cell-background", NULL, NULL);
    }
}

// Cell data function for highlighting current tool
static void text_cell_data_func(GtkTreeViewColumn *col,
                                GtkCellRenderer *renderer,
                                GtkTreeModel *model,
                                GtkTreeIter *iter,
                                gpointer user_data) {
    gboolean is_current;

    gtk_tree_model_get(model, iter, 6, &is_current, -1);

    if (is_current) {
        g_object_set(renderer, "cell-background", "#90EE90", NULL);
        g_object_set(renderer, "weight", PANGO_WEIGHT_BOLD, NULL);
    } else {
        g_object_set(renderer, "cell-background", NULL, NULL);
        g_object_set(renderer, "weight", PANGO_WEIGHT_NORMAL, NULL);
    }
}

// Create the UI
static void create_ui(MonitorApp *app) {
    GtkWidget *vbox, *hbox, *scrolled, *status_box;
    GtkWidget *db_label, *refresh_button;
    GtkCellRenderer *renderer;
    GtkTreeViewColumn *column;
    char markup[512];

    // Main vertical box
    vbox = gtk_box_new(GTK_ORIENTATION_VERTICAL, 6);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), 10);
    gtk_container_add(GTK_CONTAINER(app->window), vbox);

    // Header box
    hbox = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 10);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

    // Database file label
    snprintf(markup, sizeof(markup), "<b>Database:</b> %s", app->db_file);
    db_label = gtk_label_new(NULL);
    gtk_label_set_markup(GTK_LABEL(db_label), markup);
    gtk_label_set_xalign(GTK_LABEL(db_label), 0);
    gtk_box_pack_start(GTK_BOX(hbox), db_label, FALSE, FALSE, 0);

    // Spacer
    gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(""), TRUE, TRUE, 0);

    // Current tool label
    app->current_tool_label = gtk_label_new(NULL);
    gtk_label_set_markup(GTK_LABEL(app->current_tool_label), "<b>Current Tool:</b> None");
    gtk_box_pack_start(GTK_BOX(hbox), app->current_tool_label, FALSE, FALSE, 0);

    // Update time label
    app->update_label = gtk_label_new(NULL);
    gtk_label_set_markup(GTK_LABEL(app->update_label), "<i>Last update: Never</i>");
    gtk_label_set_xalign(GTK_LABEL(app->update_label), 1);
    gtk_box_pack_start(GTK_BOX(hbox), app->update_label, FALSE, FALSE, 0);

    // Scrolled window for tree view
    scrolled = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);

    // Create list store: tool_no, pocket, z_offset, diameter, usage, comment, is_current
    app->store = gtk_list_store_new(7, G_TYPE_INT, G_TYPE_INT, G_TYPE_DOUBLE,
                                    G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_STRING,
                                    G_TYPE_BOOLEAN);

    // Create tree view
    app->tree_view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(app->store));
    gtk_tree_view_set_grid_lines(GTK_TREE_VIEW(app->tree_view), GTK_TREE_VIEW_GRID_LINES_BOTH);
    gtk_container_add(GTK_CONTAINER(scrolled), app->tree_view);

    // Column: Tool
    renderer = gtk_cell_renderer_text_new();
    column = gtk_tree_view_column_new_with_attributes("Tool", renderer, "text", 0, NULL);
    gtk_tree_view_column_set_min_width(column, 60);
    gtk_tree_view_column_set_resizable(column, TRUE);
    gtk_tree_view_column_set_sort_column_id(column, 0);
    gtk_tree_view_column_set_cell_data_func(column, renderer, text_cell_data_func, NULL, NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(app->tree_view), column);

    // Column: Pocket
    renderer = gtk_cell_renderer_text_new();
    column = gtk_tree_view_column_new_with_attributes("Pocket", renderer, "text", 1, NULL);
    gtk_tree_view_column_set_min_width(column, 60);
    gtk_tree_view_column_set_resizable(column, TRUE);
    gtk_tree_view_column_set_sort_column_id(column, 1);
    gtk_tree_view_column_set_cell_data_func(column, renderer, text_cell_data_func, NULL, NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(app->tree_view), column);

    // Column: Z Offset
    renderer = gtk_cell_renderer_text_new();
    g_object_set(renderer, "xalign", 1.0, NULL);
    column = gtk_tree_view_column_new_with_attributes("Z Offset", renderer, NULL);
    gtk_tree_view_column_set_min_width(column, 100);
    gtk_tree_view_column_set_resizable(column, TRUE);
    gtk_tree_view_column_set_sort_column_id(column, 2);
    gtk_tree_view_column_set_cell_data_func(column, renderer, number_cell_data_func,
                                           GINT_TO_POINTER(2), NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(app->tree_view), column);

    // Column: Diameter
    renderer = gtk_cell_renderer_text_new();
    g_object_set(renderer, "xalign", 1.0, NULL);
    column = gtk_tree_view_column_new_with_attributes("Diameter", renderer, NULL);
    gtk_tree_view_column_set_min_width(column, 100);
    gtk_tree_view_column_set_resizable(column, TRUE);
    gtk_tree_view_column_set_sort_column_id(column, 3);
    gtk_tree_view_column_set_cell_data_func(column, renderer, number_cell_data_func,
                                           GINT_TO_POINTER(3), NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(app->tree_view), column);

    // Column: Usage (min)
    renderer = gtk_cell_renderer_text_new();
    g_object_set(renderer, "xalign", 1.0, NULL);
    column = gtk_tree_view_column_new_with_attributes("Usage (min)", renderer, NULL);
    gtk_tree_view_column_set_min_width(column, 120);
    gtk_tree_view_column_set_resizable(column, TRUE);
    gtk_tree_view_column_set_sort_column_id(column, 4);
    gtk_tree_view_column_set_cell_data_func(column, renderer, number_cell_data_func,
                                           GINT_TO_POINTER(4), NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(app->tree_view), column);

    // Column: Comment
    renderer = gtk_cell_renderer_text_new();
    column = gtk_tree_view_column_new_with_attributes("Comment", renderer, "text", 5, NULL);
    gtk_tree_view_column_set_min_width(column, 300);
    gtk_tree_view_column_set_resizable(column, TRUE);
    gtk_tree_view_column_set_sort_column_id(column, 5);
    gtk_tree_view_column_set_cell_data_func(column, renderer, text_cell_data_func, NULL, NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(app->tree_view), column);

    // Sort by tool number by default
    gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(app->store), 0, GTK_SORT_ASCENDING);

    // Status box
    status_box = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 10);
    gtk_box_pack_start(GTK_BOX(vbox), status_box, FALSE, FALSE, 0);

    // Status label
    app->status_label = gtk_label_new(NULL);
    gtk_label_set_markup(GTK_LABEL(app->status_label), "<i>Ready</i>");
    gtk_label_set_xalign(GTK_LABEL(app->status_label), 0);
    gtk_box_pack_start(GTK_BOX(status_box), app->status_label, TRUE, TRUE, 0);

    // Refresh button
    refresh_button = gtk_button_new_with_label("Refresh Now");
    g_signal_connect(refresh_button, "clicked", G_CALLBACK(on_refresh_clicked), app);
    gtk_box_pack_start(GTK_BOX(status_box), refresh_button, FALSE, FALSE, 0);
}

// Update data callback
static gboolean update_data(gpointer data) {
    MonitorApp *app = (MonitorApp *)data;
    struct stat st;
    int file_changed = 0;

    // Check if file was modified
    if (stat(app->db_file, &st) == 0) {
        if (st.st_mtime != app->last_mtime) {
            app->last_mtime = st.st_mtime;
            file_changed = 1;

            // Re-parse database (this also updates current_tool and current_tool_load_time)
            parse_database(app);

            // Update load_time for current tool (from database header)
            for (int i = 0; i < app->tool_count; i++) {
                if (app->tools[i].tool_no == app->current_tool) {
                    app->tools[i].load_time = app->current_tool_load_time;
                } else {
                    app->tools[i].load_time = 0;
                }
            }
        }
    }

    // Always update display for live elapsed time calculation
    // But if file didn't change and no current tool, we can skip
    if (file_changed || app->current_tool > 0) {
        update_display(app);
    }

    return TRUE;  // Continue periodic updates
}

// Refresh button callback
static void on_refresh_clicked(GtkWidget *widget, gpointer data) {
    update_data(data);
}

int main(int argc, char *argv[]) {
    MonitorApp app;
    char *home;

    // Initialize
    memset(&app, 0, sizeof(app));
    app.current_tool = -1;
    app.prev_current_tool = -1;
    app.updating = 0;

    // Get database file from command line or use default
    if (argc > 1) {
        strncpy(app.db_file, argv[1], sizeof(app.db_file) - 1);
    } else {
        home = getenv("HOME");
        if (home) {
            snprintf(app.db_file, sizeof(app.db_file),
                    "%s/linuxcnc_tool_data/tool_usage.dat", home);
        } else {
            fprintf(stderr, "Error: Cannot determine home directory\n");
            fprintf(stderr, "Usage: %s <database_file>\n", argv[0]);
            return 1;
        }
    }

    // Check if database exists
    if (access(app.db_file, R_OK) != 0) {
        fprintf(stderr, "Error: Database file not found: %s\n", app.db_file);
        fprintf(stderr, "Usage: %s [database_file]\n", argv[0]);
        fprintf(stderr, "Example: %s ~/linuxcnc_tool_data/tool_usage.dat\n", argv[0]);
        return 1;
    }

    // Initialize GTK
    gtk_init(&argc, &argv);

    // Create window
    app.window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(app.window), "LinuxCNC Tool Usage Monitor");
    gtk_window_set_default_size(GTK_WINDOW(app.window), 1200, 400);
    g_signal_connect(app.window, "destroy", G_CALLBACK(gtk_main_quit), NULL);

    // Create UI
    create_ui(&app);

    // Load initial data (this also reads current_tool and current_tool_load_time from database)
    parse_database(&app);

    // Override database info with actual LinuxCNC state on startup
    time_t detected_load_time = 0;
    int detected_tool = detect_current_tool(&detected_load_time);

    if (detected_tool >= 0) {
        // Use detected tool (could be 0 if no tool loaded)
        app.current_tool = detected_tool;
        app.current_tool_load_time = detected_load_time;
        fprintf(stderr, "Detected current tool from LinuxCNC: T%d\n", detected_tool);
    } else {
        fprintf(stderr, "Could not detect current tool, using database value: T%d\n", app.current_tool);
    }

    // Set load_time for current tool
    for (int i = 0; i < app.tool_count; i++) {
        if (app.tools[i].tool_no == app.current_tool) {
            app.tools[i].load_time = app.current_tool_load_time;
        } else {
            app.tools[i].load_time = 0;
        }
    }

    update_display(&app);

    // Start periodic updates (every 1 second = 1000ms)
    g_timeout_add(1000, update_data, &app);

    // Show window
    gtk_widget_show_all(app.window);

    // Run GTK main loop
    gtk_main();

    return 0;
}
